import React, { MouseEvent, useEffect, useState } from 'react';
import { Provider } from 'react-redux';
import {
    DotAvatar,
    DotBadge,
    DotButton,
    DotChip,
    DotIcon,
    DotIconButton,
    DotInputText,
    DotTabs,
    DotThemeProvider,
    DotTypography,
} from '@digital-ai/dot-components';
import { store } from '../../analytics-store';
import { useAppDispatch, useAppSelector } from '@xlr-ui/app/js/hooks';
import { useInfiniteScrollPagination } from '@xlr-ui/app/features/common/hooks/infinite-scroll-pagination.hook';
import { getLicense } from '@xlr-ui/app/features/license/ducks/license-warning.selectors';
import { Folder, License } from '@xlr-ui/app/types';
import { AnalyticsCards } from './analytics-cards.component';
import { analytics, AnalyticsPageFilter, AnalyticsTabsEnum, CardSearch, getAnalyticsState } from '../ducks/analytics.reducer';
import { calculateNumberOfFiltersApplied, removeAuthorFromFilters, removeCategoryFromFilters } from '../helpers';
import { AnalyticsPageFilterDrawer } from './analytics-page-filter-drawer.component';
import { AnalyticsCardsSkeleton } from './analytics-cards.skeleton';
import { AnalyticsPagePortal } from './analytics-page.portal';
import { AnalyticsCardData } from '../types';
import { DEFAULT_PAGINATION } from '@xlr-ui/app/constants/pagination';
import './analytics-page.component.less';

export interface AnalyticsPageProps {
    filters: AnalyticsPageFilter;
    folder?: Folder;
}

export const AnalyticsPageProvider = (props: AnalyticsPageProps) => {
    return (
        <Provider store={store}>
            <DotThemeProvider>
                <AnalyticsPage {...props} />
            </DotThemeProvider>
        </Provider>
    );
};

const ANALYTICS_PAGE_TABS = [{ label: 'All dashboards' }, { label: 'Favorites' }];

const { init, filterCards, setIsDrawerOpened, favoriteDashboardFromList, resetAnalytics, setSelectedTab } = analytics.actions;

const LICENSE_EDITIONS_FOR_BANNER = ['Essentials', 'Pro', 'Premium', 'Trial'];

export const AnalyticsPage = ({ filters, folder }: AnalyticsPageProps) => {
    const dispatch = useAppDispatch();
    const [currentCards, setCurrentCards] = useState<AnalyticsCardData[]>([]);
    const license: License | undefined = useAppSelector(getLicense);
    const {
        authors,
        isDrawerOpened,
        allCards,
        favoriteCards,
        categories,
        cardSearch,
        intelligenceConfiguration,
        isCategoriesLoading,
        isLoading,
        selectedTab,
        totalAvailableCards,
        totalAvailableFavoriteCards,
        manageDashboardUrl,
        updatingDashboardIds,
    } = useAppSelector(getAnalyticsState);
    const isAnalyticsConfigured = !!intelligenceConfiguration;
    const isBannerVisible = LICENSE_EDITIONS_FOR_BANNER.includes(license?.edition || 'Community') && !isAnalyticsConfigured;
    const numberOfFiltersApplied = calculateNumberOfFiltersApplied(cardSearch);
    const {
        categories: filteredCategories,
        authorNames: filteredAuthors,
        page = DEFAULT_PAGINATION.page,
        itemsPerPage = DEFAULT_PAGINATION.itemsPerPage,
    } = cardSearch || {};

    const isAnyCategoryFilterPresent = !!(filteredCategories && filteredCategories.length > 0);
    const isAnyAuthorFilterPresent = !!(filteredAuthors && filteredAuthors.length > 0);
    const isAnyFilterPresent = isAnyAuthorFilterPresent || isAnyCategoryFilterPresent;
    const isSearchOrFilterApplied = (cardSearch && cardSearch.name !== '') || isAnyFilterPresent;
    const isInitialLoad = !currentCards.length && isLoading;
    const isFavoritesTab = selectedTab === AnalyticsTabsEnum.Favorites;
    const totalTabCards = isFavoritesTab ? totalAvailableFavoriteCards : totalAvailableCards;
    const totalPages = Math.ceil(totalTabCards / itemsPerPage);
    const isLastPage = page + 1 >= totalPages;

    useEffect(() => {
        dispatch(init(filters));
        return () => dispatch(resetAnalytics());
    }, []);

    useEffect(() => {
        if (isLoading) return;
        const cards = isFavoritesTab ? favoriteCards : allCards;
        setCurrentCards(cards);
    }, [allCards, favoriteCards, selectedTab, isLoading]);

    const getCardSearch = (currentCardSearch: CardSearch) => (cardSearch ? { ...cardSearch, ...currentCardSearch } : currentCardSearch);
    const handleDrawerClose = () => dispatch(setIsDrawerOpened(false));
    const handleCardFilterChange = (currentCardSearch: CardSearch) => {
        currentCardSearch.page === 0 && setCurrentCards([]);
        dispatch(filterCards(currentCardSearch));
    };
    const handleTabChange = (value: number) => {
        dispatch(setSelectedTab(value));
    };
    const handleClearAllFilters = () =>
        handleCardFilterChange(
            getCardSearch({
                ...cardSearch,
                ...defaultPagination,
                categories: undefined,
                authorNames: undefined,
            }),
        );

    const handleDrawerOpen = (event: MouseEvent<HTMLButtonElement>) => {
        if (isDrawerOpened) return;
        event.stopPropagation();
        dispatch(setIsDrawerOpened(true));
    };
    const handleIntersect = () => {
        if (isLastPage) return;
        handleCardFilterChange(getCardSearch({ page: page + 1, itemsPerPage, isFavorite: isFavoritesTab }));
    };

    const handleFavoriteDashboard = (id: string, isCurrentFavorite: boolean) => {
        dispatch(favoriteDashboardFromList({ id, isUnfavorite: isCurrentFavorite }));
    };

    const { observerTarget, defaultPagination } = useInfiniteScrollPagination<AnalyticsCardData, HTMLDivElement>({
        items: currentCards,
        itemsPerPage,
        onIntersect: handleIntersect,
        page,
        totalPages,
    });

    const getAnalyticsTabsWithNumbers = () => {
        return ANALYTICS_PAGE_TABS.map((tab, index) => {
            const tabCount = index === 0 ? totalAvailableCards : totalAvailableFavoriteCards;
            return { label: `${tab.label} (${tabCount})` };
        });
    };

    const renderCardsSkeleton = () => {
        return (
            <>
                {selectedTab === AnalyticsTabsEnum.AllDashboards && <AnalyticsCardsSkeleton />}
                {selectedTab === AnalyticsTabsEnum.Favorites && <AnalyticsCardsSkeleton />}
            </>
        );
    };

    const renderAnalyticsCards = () => {
        return (
            <AnalyticsCards
                cards={currentCards}
                folderId={folder?.id}
                isAnalyticsConfigured={isAnalyticsConfigured}
                isFavoritePage={isFavoritesTab}
                isLoading={isLoading}
                isSearchOrFilterApplied={isSearchOrFilterApplied}
                onFavoriteDashboard={handleFavoriteDashboard}
                updatingDashboardIds={updatingDashboardIds}
            />
        );
    };

    const renderPage = () => {
        return (
            <div className="analytics-page">
                <AnalyticsPagePortal buttonHref={manageDashboardUrl} />
                <div className="analytics-page-content">
                    {isBannerVisible && (
                        <div className="analytics-banner">
                            <DotAvatar alt="crown icon" className="analytics-banner-avatar" iconId="crown" />
                            <DotTypography variant="body1">
                                Analytics dashboards are available in the Premium edition. For more information, please contact your Customer Success Manager.
                            </DotTypography>
                        </div>
                    )}
                    <div className="content-header">
                        <DotTypography data-testid="page-title" variant="h1">
                            Analytics
                        </DotTypography>
                        <div className="content-header-filter">
                            <DotInputText
                                className="name-filter"
                                defaultValue={filters.name}
                                hasDebounce={true}
                                id="name-filter"
                                name="name-filter"
                                onChange={(e) =>
                                    handleCardFilterChange(getCardSearch({ name: e.target.value, ...defaultPagination, isFavorite: isFavoritesTab }))
                                }
                                onClear={() => handleCardFilterChange(getCardSearch({ name: '', ...defaultPagination, isFavorite: isFavoritesTab }))}
                                placeholder="Filter by name ..."
                                startIcon={<DotIcon data-testid="name-filter-search-icon" fontSize="small" iconId="search" />}
                            />
                            <DotBadge badgeColor="#d61f21" badgeContent={numberOfFiltersApplied} overlap="circular" variant="standard">
                                <DotIconButton iconId="filter" onClick={handleDrawerOpen} />
                            </DotBadge>
                        </div>
                    </div>

                    {isAnyFilterPresent && (
                        <div className="aligned-flex-with-gap" data-testid="filters-row">
                            {filteredCategories && filteredCategories.length > 0 && (
                                <>
                                    <DotTypography data-testid="category-filters-title" variant="subtitle2">
                                        Categories:
                                    </DotTypography>
                                    {filteredCategories.map((categoryName) => (
                                        <DotChip
                                            data-testid={`category-filters-${categoryName}`}
                                            key={categoryName}
                                            onDelete={() =>
                                                handleCardFilterChange(
                                                    getCardSearch({
                                                        categories: removeCategoryFromFilters(categoryName, filteredCategories),
                                                        ...defaultPagination,
                                                        isFavorite: isFavoritesTab,
                                                    }),
                                                )
                                            }
                                            size="small"
                                        >
                                            {categoryName}
                                        </DotChip>
                                    ))}
                                </>
                            )}
                            {filteredAuthors && filteredAuthors.length > 0 && (
                                <>
                                    <DotTypography data-testid="author-filters-title" variant="subtitle2">
                                        Authors:
                                    </DotTypography>
                                    {filteredAuthors.map((authorName) => (
                                        <DotChip
                                            avatar={<DotAvatar alt={`Image for user ${authorName}`} size="small" text={authorName} type="text" />}
                                            data-testid={`author-filters-${authorName}`}
                                            key={authorName}
                                            onDelete={() =>
                                                handleCardFilterChange(
                                                    getCardSearch({
                                                        authorNames: removeAuthorFromFilters(authorName, filteredAuthors),
                                                        isFavorite: isFavoritesTab,
                                                        ...defaultPagination,
                                                    }),
                                                )
                                            }
                                            size="small"
                                        >
                                            {authorName}
                                        </DotChip>
                                    ))}
                                </>
                            )}
                            <DotButton className="chipy-btn" data-testid="clear-all-btn" onClick={handleClearAllFilters} type="text">
                                Clear all
                            </DotButton>
                        </div>
                    )}
                    <DotTabs
                        className="analytics-tabs"
                        initialValue={selectedTab}
                        onChange={handleTabChange}
                        tabs={isLoading ? ANALYTICS_PAGE_TABS : getAnalyticsTabsWithNumbers()}
                    />
                    <div className="analytics-tabs-wrapper">{isInitialLoad ? renderCardsSkeleton() : renderAnalyticsCards()}</div>
                </div>
                {currentCards && !isLoading && <div ref={observerTarget} />}
            </div>
        );
    };

    return (
        <>
            {renderPage()}
            <AnalyticsPageFilterDrawer
                authors={authors}
                cardSearch={cardSearch}
                categories={categories}
                isCategoriesLoading={isCategoriesLoading}
                isDrawerOpened={isDrawerOpened}
                numberOfFiltersApplied={numberOfFiltersApplied}
                onCardFilter={handleCardFilterChange}
                onClearAll={handleClearAllFilters}
                onDrawerClose={handleDrawerClose}
            />
        </>
    );
};
