import { SagaIterator } from 'redux-saga';
import { all, call, put, select, takeEvery, takeLatest } from 'redux-saga/effects';
import {
    analyticsSaga,
    checkIfIntelligenceIsConfigured,
    doPostWithToken,
    favoriteDashboardFromDetailsSaga,
    favoriteDashboardFromListSaga,
    fetchDashboardsFromApi,
    filterCardsSaga,
    getToken,
    initSaga,
    loadAnalyticsCardsData,
    loadAnalyticsCardsForBothTabs,
    loadDashboardSaga,
    loadFavoriteCardsSaga,
    loadMetadata,
    sendFavoriteDashboardRequest,
    setCardSearchSaga,
    setFilters,
    setManageDashboardUrlIfUserHasRole,
    toaster,
} from './analytics.saga';
import { analytics, CardSearch, getAnalyticsState } from './analytics.reducer';
import { ANALYTICS_AUTHORS, ANALYTICS_CATEGORIES, DIGITAL_AI_AUTHOR, INTELLIGENCE_SERVER_TITLE } from '../constants';
import getAngularService from '@xlr-ui/app/features/common/services/angular-accessor';
import { httpGET, httpPOST } from '@xlr-ui/app/features/common/services/http';
import { DEFAULT_PAGINATION } from '@xlr-ui/app/constants/pagination';
import { AnalyticsCardData, IntelligenceHttpConnection, DashboardView } from '../types';
import { getAuthorName, getFilteredSampleCards, mapSupersetToAnalyticsCard, mapSupersetToAnalyticsCards } from '../helpers';

const {
    addToUpdatingDashboardIds,
    favoriteDashboardFromDetails,
    favoriteDashboardFromList,
    filterCards,
    init,
    loadDashboard,
    loadFavoriteCards,
    refreshToken,
    removeFromFavoritesCards,
    removeFromUpdatingDashboardIds,
    replaceCardInAllCards,
    setAllCards,
    setAuthors,
    setCardSearch,
    setCategories,
    setFavoriteCards,
    setIntelligenceConfiguration,
    setIsCategoriesLoading,
    setIsLoading,
    setManageDashboardUrl,
    setSelectedDashboard,
    setTokenData,
    setTotalAvailableCards,
    setTotalAvailableFavoriteCards,
} = analytics.actions;

describe('analyticsSaga', () => {
    const defaultPagination = DEFAULT_PAGINATION;
    const cards: AnalyticsCardData[] = [
        {
            contentText: 'Website Traffic Report for October',
            id: 'card1',
            imageSrc: 'test',
            isFavorite: false,
            subtitle: DIGITAL_AI_AUTHOR,
            tags: [ANALYTICS_CATEGORIES[0], ANALYTICS_CATEGORIES[1], ANALYTICS_CATEGORIES[2]],
            title: 'Onboarding plan and progress A',
        },
        {
            contentText: 'Website Traffic Report for December',
            id: 'card1',
            imageSrc: 'test',
            isFavorite: false,
            subtitle: 'Custom',
            tags: [ANALYTICS_CATEGORIES[1]],
            title: 'Onboarding plan and progress B',
        },
    ] as never;
    const favoriteCards: AnalyticsCardData[] = [
        {
            contentText: 'Test favorite card',
            id: 'fav-card-1',
            imageSrc: 'test',
            isFavorite: true,
            subtitle: 'Custom',
            tags: [ANALYTICS_CATEGORIES[1]],
            title: 'Onboarding plan and progress B',
        },
    ] as never;

    const response = {
        entity: {
            results: [
                {
                    id: 1,
                    thumbnail: 'thumbnail',
                    external_id: '1',
                    filter_configuration: [
                        {
                            summary: 'Tags In List (All)',
                            name: 'Tags',
                            source: {
                                id: '941166EDD92F4BDDBB264D9DEDFE9D2E',
                                type: 12,
                            },
                            key: 'W8E2C0BF756EF4EBC9F04607BC5285770',
                        },
                        {
                            summary: '',
                            name: 'Folder',
                            source: {
                                id: 'EFC67C796B6949D0BD02630332B0A98D',
                                type: 12,
                            },
                            key: 'W1C5B072BE1174884905C60DDEC2DEB79',
                        },
                        {
                            summary: '',
                            name: 'User',
                            source: {
                                id: '7C80778BC1B5484388B645DD7F6F2FAB',
                                type: 12,
                            },
                            key: 'W6B8B4F8CCB1D42DDA736A8BF406CC40D',
                        },
                        {
                            summary: "(([Workflow start date]@ID Between '4/6/2024' And '5/6/2024 11:59:59 PM'))",
                            name: 'Start date',
                            source: {
                                id: '3D1555170D1542DF9D733E047317ACB3',
                                type: 12,
                            },
                            key: 'WF4E7E5C872D34C79B9DD745FB09416BF',
                        },
                        {
                            summary: '',
                            name: 'Team',
                            source: {
                                id: '8DB0515AC64246709B0BE907393611AB',
                                type: 12,
                            },
                            key: 'WBA087DF0CB2144D88EDEF312676DFAE8',
                        },
                        {
                            summary: 'Workflow category In List (All)',
                            name: 'Workflow category',
                            source: {
                                id: '25BC07D902C94923ADC6FA55A4F1A569',
                                type: 12,
                            },
                            key: 'W699239EE571249E88F678D827B5489E0',
                        },
                    ],
                    author_fullname: 'author',
                    categories: ['DevOps COE'],
                },
            ],
            pagination: {
                total_available: 1,
            },
        },
    };

    const favoriteEntity: DashboardView = {
        application_instances: [],
        author_id: '',
        bi_type: DashboardView.bi_type.MICROSTRATEGY,
        dashboard_url: '',
        external_object: {},
        help_content_id: '',
        target_apps: ['RELEASE'],
        favorite: false,
        description: '',
        external_embedding_id: '',
        thumbnail: 'abc',
        id: '66447280bcc3f4d5bb396a94',
        lifecycle_state: DashboardView.lifecycle_state.DRAFT,
        external_id: '246DA7F4864D5EF7D248528B6750DAC0',
        server_url: 'https://intelligence-devops.bi.staging.digital.ai',
        filter_configuration: [],
        name: 'release_test',
        author_fullname: 'Surabhi Sundhar',
        is_ootb_dashboard: false,
        embed_config: {} as never,
        categories: [],
        created_dt: '2024-05-15T08:29:50.841000',
    };

    const mappedFavoriteEntity = {
        embeddingId: favoriteEntity.external_embedding_id,
        embedConfig: favoriteEntity.embed_config,
        id: favoriteEntity.id,
        isFavorite: favoriteEntity.favorite,
        externalId: parseInt(favoriteEntity.external_id),
        tags: favoriteEntity.categories,
        title: favoriteEntity.name,
        serverUrl: favoriteEntity.server_url,
        subtitle: getAuthorName(favoriteEntity),
        imageSrc: favoriteEntity.thumbnail,
        contentText: favoriteEntity.description,
    } as never;

    it('should yield all effects', () => {
        const gen: SagaIterator = analyticsSaga();
        expect(gen.next().value).toStrictEqual(
            all([
                takeLatest(init, initSaga),
                takeLatest(filterCards, filterCardsSaga),
                takeEvery(favoriteDashboardFromDetails, favoriteDashboardFromDetailsSaga),
                takeEvery(favoriteDashboardFromList, favoriteDashboardFromListSaga),
                takeLatest(setCardSearch, setCardSearchSaga),
                takeLatest(loadFavoriteCards, loadFavoriteCardsSaga),
                takeLatest(loadDashboard, loadDashboardSaga),
                takeLatest(refreshToken, getToken),
            ]),
        );
        expect(gen.next().done).toBe(true);
    });

    describe('initSaga', () => {
        const cardSearch = { name: 'abc', isFavorite: false, ...defaultPagination };

        it('should call proper saga', () => {
            const gen: SagaIterator = initSaga({ payload: cardSearch, type: 'test' });
            expect(gen.next().value).toStrictEqual(put(setIsLoading(true)));
            expect(gen.next().value).toStrictEqual(put(setAllCards([])));
            expect(gen.next().value).toStrictEqual(call(checkIfIntelligenceIsConfigured));
            expect(gen.next().value).toStrictEqual(call(loadMetadata));
            expect(gen.next().value).toStrictEqual(call(setFilters, cardSearch));
            expect(gen.next().value).toStrictEqual(all([call(loadAnalyticsCardsData), call(loadAnalyticsCardsData, true)]));
            expect(
                gen.next([
                    [cards, 2],
                    [favoriteCards, 1],
                ]).value,
            ).toStrictEqual(put(setIsLoading(false)));
            expect(gen.next().value).toStrictEqual(put(setFavoriteCards(favoriteCards)));
            expect(gen.next().value).toStrictEqual(put(setTotalAvailableFavoriteCards(1)));
            expect(gen.next().value).toStrictEqual(put(setAllCards(cards)));
            expect(gen.next().value).toStrictEqual(put(setTotalAvailableCards(2)));
            expect(gen.next().value).toStrictEqual(call(setManageDashboardUrlIfUserHasRole));
            expect(gen.next().done).toBe(true);
        });
    });

    describe('loadAnalyticsCardsForBothTabs', () => {
        it('should call proper saga', () => {
            const gen: SagaIterator = loadAnalyticsCardsForBothTabs();
            expect(gen.next().value).toStrictEqual(all([call(loadAnalyticsCardsData), call(loadAnalyticsCardsData, true)]));
            expect(
                gen.next([
                    [cards, 2],
                    [favoriteCards, 1],
                ]).value,
            ).toStrictEqual(put(setFavoriteCards(favoriteCards)));
            expect(gen.next().value).toStrictEqual(put(setTotalAvailableFavoriteCards(1)));
            expect(gen.next().value).toStrictEqual(put(setAllCards(cards)));
            expect(gen.next().value).toStrictEqual(put(setTotalAvailableCards(2)));
            expect(gen.next().done).toBe(true);
        });
    });

    describe('setManageDashboardUrlIfUserHasRole', () => {
        const intelligenceConfiguration = {
            id: '1',
            url: 'https://vanity.region.digital.ai/',
        };

        const runSaga = (): SagaIterator => {
            const gen: SagaIterator = setManageDashboardUrlIfUserHasRole();
            expect(gen.next().value).toStrictEqual(select(getAnalyticsState));
            expect(
                gen.next({
                    intelligenceConfiguration,
                }).value,
            ).toStrictEqual(call(doPostWithToken, `api/extension/intelligence/user?serverId=${intelligenceConfiguration.id}`));
            return gen;
        };

        it('should call proper sagas when user does not have proper role', () => {
            const user = {
                roles: [],
            };

            const gen: SagaIterator = runSaga();
            expect(gen.next({ entity: user }).done).toBe(true);
        });
        it('should call proper sagas when user has proper role', () => {
            const user = {
                account_id: 'my_account_id',
                roles: ['account-analytics-author'],
                account: {
                    vanity_domain: 'intelligence-devops',
                },
            };

            const gen: SagaIterator = runSaga();
            expect(gen.next({ entity: user }).value).toStrictEqual(
                put(setManageDashboardUrl('https://intelligence-devops.region.digital.ai/accounts/my_account_id/analytics/dashboards?app=RELEASE')),
            );
            expect(gen.next().done).toBe(true);
        });
    });

    describe('checkIfIntelligenceIsConfigured', () => {
        const configurationService = {
            searchConfiguration: jest.fn(),
        };

        it('should set analytics configured to true if there is intelligence server', () => {
            const gen: SagaIterator = checkIfIntelligenceIsConfigured();
            expect(gen.next().value).toStrictEqual(call(getAngularService, 'ConfigurationService'));
            expect(gen.next(configurationService).value).toStrictEqual(
                call([configurationService, configurationService.searchConfiguration], 'dai.IntelligenceServer', INTELLIGENCE_SERVER_TITLE),
            );
            expect(gen.next({ 'dai.IntelligenceServer': [{ id: 'FakeServer' }] }).value).toStrictEqual(
                put(setIntelligenceConfiguration({ id: 'FakeServer' } as never)),
            );
        });

        it('should set analytics configured to false if there is no intelligence server', () => {
            const gen: SagaIterator = checkIfIntelligenceIsConfigured();
            expect(gen.next().value).toStrictEqual(call(getAngularService, 'ConfigurationService'));
            expect(gen.next(configurationService).value).toStrictEqual(
                call([configurationService, configurationService.searchConfiguration], 'dai.IntelligenceServer', INTELLIGENCE_SERVER_TITLE),
            );
            expect(gen.next({ 'dai.IntelligenceServer': [] }).value).toStrictEqual(put(setIntelligenceConfiguration(undefined)));
        });
    });

    describe('loadMetadata', () => {
        const categories = ['Category 2', 'Category 1'];
        const authors = [
            {
                full_name: 'John Caldas',
                id: '2ddadfce-7640-477a-a39e-6ac1f28b514d',
            },
            {
                full_name: ' ',
                id: '21156748-a5b2-4d60-b35f-6ba13d3306e9',
            },
            {
                full_name: 'Anna Smith',
                id: '111111-a5b2-4d60-b35f-22213d3306e4',
            },
            {
                full_name: 'Anna Smith',
                id: '333333-a5b2-4d60-b35f-22213d3306e5',
            },
        ];
        const metadataResponse = {
            entity: {
                categories,
                authors,
            },
        };

        it('should load metadata from the api', () => {
            const state = {
                intelligenceConfiguration: { id: 'MyServerId', folderColumnName: 'Folder' },
            };

            const gen: SagaIterator = loadMetadata();
            expect(gen.next().value).toStrictEqual(select(getAnalyticsState));
            expect(gen.next(state).value).toStrictEqual(put(setIsCategoriesLoading(true)));
            expect(gen.next().value).toStrictEqual(call(doPostWithToken, `api/extension/intelligence/metadata?serverId=${state.intelligenceConfiguration.id}`));
            expect(gen.next(metadataResponse).value).toStrictEqual(put(setCategories(['Category 1', 'Category 2'])));
            expect(gen.next().value).toStrictEqual(put(setAuthors([DIGITAL_AI_AUTHOR, 'Anna Smith', 'Anna Smith', 'John Caldas'])));
            expect(gen.next().value).toStrictEqual(put(setIsCategoriesLoading(false)));
            expect(gen.next().done).toBe(true);
        });

        it('should set sample categories and authors array when intelligence is not configured', () => {
            const state = {
                intelligenceConfiguration: undefined,
            };

            const gen: SagaIterator = loadMetadata();
            expect(gen.next().value).toStrictEqual(select(getAnalyticsState));
            expect(gen.next(state).value).toStrictEqual(put(setIsCategoriesLoading(true)));
            expect(gen.next().value).toStrictEqual(put(setCategories(ANALYTICS_CATEGORIES)));
            expect(gen.next().value).toStrictEqual(put(setAuthors(ANALYTICS_AUTHORS)));
            expect(gen.next().value).toStrictEqual(put(setIsCategoriesLoading(false)));
            expect(gen.next().done).toBe(true);
        });
    });

    describe('fetchDashboardsFromApi', () => {
        const intelligenceConfigurationId = '123';
        const cardSearch: CardSearch = { name: 'abc', isFavorite: false, ...defaultPagination };

        it('should fetch dashboard data from the API', () => {
            const gen: SagaIterator = fetchDashboardsFromApi(intelligenceConfigurationId, cardSearch);
            expect(gen.next().value).toStrictEqual(
                call(
                    doPostWithToken,
                    `api/extension/intelligence/dashboards?serverId=${intelligenceConfigurationId}&page=${defaultPagination.page}&resultsPerPage=${defaultPagination.itemsPerPage}&filter=name[isearch]:abc`,
                ),
            );
            expect(gen.next(response)).toStrictEqual({ done: true, value: response.entity });
        });
    });

    describe('loadAnalyticsCardsData', () => {
        it('should load analytics cards data', () => {
            const folderColumnName = 'Folder';
            const state = {
                intelligenceConfiguration: { id: 'MyServerId', folderColumnName },
                cardSearch: { ...defaultPagination },
            };
            const entity = response.entity;

            const gen: SagaIterator = loadAnalyticsCardsData(false);
            expect(gen.next().value).toStrictEqual(select(getAnalyticsState));
            expect(gen.next(state).value).toStrictEqual(
                call(fetchDashboardsFromApi, state.intelligenceConfiguration.id, { ...state.cardSearch, isFavorite: false }),
            );
            expect(gen.next(entity)).toStrictEqual({
                done: true,
                value: [mapSupersetToAnalyticsCards(entity.results as never, folderColumnName), entity.pagination.total_available],
            });
        });

        it('should load default cards when intelligence configuration is not defined', () => {
            const cardSearch: CardSearch = { ...defaultPagination, name: 'Efficiently monitor' };
            const state = {
                intelligenceConfiguration: null,
                cardSearch,
            };

            const filteredCards = getFilteredSampleCards(cardSearch);

            const gen: SagaIterator = loadAnalyticsCardsData(false);
            expect(gen.next().value).toStrictEqual(select(getAnalyticsState));
            expect(gen.next(state)).toStrictEqual({
                done: true,
                value: [filteredCards, filteredCards.length],
            });
        });
    });

    describe('favoriteDashboardFromListSaga', () => {
        const id = '123';
        const state = {
            intelligenceConfiguration: { id: 'MyServerId', folderColumnName: 'FOLDER_COLUMN' } as IntelligenceHttpConnection,
            cardSearch: { ...defaultPagination },
        };

        it('should not send API request when intelligence is NOT configured', () => {
            const gen: SagaIterator = favoriteDashboardFromListSaga({ payload: { id, isUnfavorite: false }, type: 'test' });
            expect(gen.next().value).toStrictEqual(select(getAnalyticsState));
            expect(gen.next({ intelligenceConfiguration: null }).done).toBe(true);
        });

        it('should favorite dashboard when intelligence is configured', () => {
            const gen: SagaIterator = favoriteDashboardFromListSaga({ payload: { id, isUnfavorite: false }, type: 'test' });
            expect(gen.next().value).toStrictEqual(select(getAnalyticsState));
            expect(gen.next(state).value).toStrictEqual(put(addToUpdatingDashboardIds(id)));
            expect(gen.next().value).toStrictEqual(call(sendFavoriteDashboardRequest, state.intelligenceConfiguration, id, false));
            expect(gen.next(mappedFavoriteEntity).value).toStrictEqual(put(loadFavoriteCards()));
            expect(gen.next().value).toStrictEqual(put(replaceCardInAllCards(mappedFavoriteEntity)));
            expect(gen.next().value).toStrictEqual(put(removeFromUpdatingDashboardIds(id)));
            expect(gen.next(response.entity).done).toBe(true);
        });

        it('should un-favorite dashboard when intelligence is configured', () => {
            const gen: SagaIterator = favoriteDashboardFromListSaga({ payload: { id, isUnfavorite: true }, type: 'test' });
            expect(gen.next().value).toStrictEqual(select(getAnalyticsState));
            expect(gen.next(state).value).toStrictEqual(put(addToUpdatingDashboardIds(id)));
            expect(gen.next().value).toStrictEqual(call(sendFavoriteDashboardRequest, state.intelligenceConfiguration, id, true));
            expect(gen.next(mappedFavoriteEntity).value).toStrictEqual(put(removeFromFavoritesCards(id)));
            expect(gen.next().value).toStrictEqual(put(replaceCardInAllCards(mappedFavoriteEntity)));
            expect(gen.next().value).toStrictEqual(put(removeFromUpdatingDashboardIds(id)));
            expect(gen.next(response.entity).done).toBe(true);
        });
    });

    describe('favoriteDashboardFromDetailsSaga', () => {
        const id = '123';
        const state = {
            intelligenceConfiguration: { id: 'MyServerId', folderColumnName: 'FOLDER_COLUMN' } as IntelligenceHttpConnection,
            cardSearch: { ...defaultPagination },
        };

        it('should not send API request when intelligence is NOT configured', () => {
            const gen: SagaIterator = favoriteDashboardFromDetailsSaga({ payload: { id, isUnfavorite: false }, type: 'test' });
            expect(gen.next().value).toStrictEqual(select(getAnalyticsState));
            expect(gen.next({ intelligenceConfiguration: null }).done).toBe(true);
        });

        it('should favorite dashboard when intelligence is configured', () => {
            const gen: SagaIterator = favoriteDashboardFromDetailsSaga({ payload: { id, isUnfavorite: false }, type: 'test' });
            expect(gen.next().value).toStrictEqual(select(getAnalyticsState));
            expect(gen.next(state).value).toStrictEqual(put(addToUpdatingDashboardIds(id)));
            expect(gen.next().value).toStrictEqual(call(sendFavoriteDashboardRequest, state.intelligenceConfiguration, id, false));
            expect(gen.next(mappedFavoriteEntity).value).toStrictEqual(put(setSelectedDashboard(mappedFavoriteEntity)));
            expect(gen.next().value).toStrictEqual(put(removeFromUpdatingDashboardIds(id)));
            expect(gen.next(response.entity).done).toBe(true);
        });

        it('should un-favorite dashboard when intelligence is configured', () => {
            const gen: SagaIterator = favoriteDashboardFromDetailsSaga({ payload: { id, isUnfavorite: true }, type: 'test' });
            expect(gen.next().value).toStrictEqual(select(getAnalyticsState));
            expect(gen.next(state).value).toStrictEqual(put(addToUpdatingDashboardIds(id)));
            expect(gen.next().value).toStrictEqual(call(sendFavoriteDashboardRequest, state.intelligenceConfiguration, id, true));
            expect(gen.next(mappedFavoriteEntity).value).toStrictEqual(put(setSelectedDashboard(mappedFavoriteEntity)));
            expect(gen.next().value).toStrictEqual(put(removeFromUpdatingDashboardIds(id)));
            expect(gen.next(response.entity).done).toBe(true);
        });
    });

    describe('sendFavoriteDashboardRequest', () => {
        const intelligenceConfiguration: IntelligenceHttpConnection = { id: 'MyServerId', folderColumnName: 'FOLDER_COLUMN' } as never;
        const dashboardId = '123';

        it('should load proper sagas when sending favorite dashboard request when dashboard is not a favorite', () => {
            const gen: SagaIterator = sendFavoriteDashboardRequest(intelligenceConfiguration, '123', false);
            expect(gen.next().value).toStrictEqual(
                call(doPostWithToken, `api/extension/intelligence/dashboards/favorite?serverId=${intelligenceConfiguration.id}&dashboardId=${dashboardId}`),
            );
            expect(gen.next(response)).toStrictEqual({
                done: true,
                value: mapSupersetToAnalyticsCard(response.entity as never, intelligenceConfiguration?.folderColumnName),
            });
        });

        it('should load proper sagas when sending favorite dashboard request when dashboard is a favorite', () => {
            const gen: SagaIterator = sendFavoriteDashboardRequest(intelligenceConfiguration, '123', true);
            expect(gen.next().value).toStrictEqual(
                call(
                    doPostWithToken,
                    `api/extension/intelligence/dashboards/favorite?serverId=${intelligenceConfiguration.id}&dashboardId=${dashboardId}&unfavorite=1`,
                ),
            );
            expect(gen.next(response).done).toBe(true);
        });
    });

    describe('loadFavoriteCardsSaga', () => {
        it('should call proper sagas', () => {
            const gen: SagaIterator = loadFavoriteCardsSaga();
            expect(gen.next().value).toStrictEqual(call(loadAnalyticsCardsData, true, 0));
            expect(gen.next([favoriteCards, 1]).value).toStrictEqual(put(setFavoriteCards(favoriteCards)));
            expect(gen.next().value).toStrictEqual(put(setTotalAvailableFavoriteCards(1)));
        });
    });

    describe('setFilters', () => {
        const analyticsTestState = { authors: ['author'], categories: ['DevOps COE'] };
        const cardSearch: CardSearch = {
            name: 'progress B',
            authorNames: ['invalid-author', 'author'],
            categories: ['invalid-category', 'DevOps COE'],
            ...defaultPagination,
        };
        const cardSearchUndefinedAuthors: CardSearch = { name: 'progress B', authorNames: undefined, categories: undefined, ...defaultPagination };

        it('should call proper sagas when authors and categories are undefined', () => {
            const gen: SagaIterator = setFilters(cardSearchUndefinedAuthors);
            expect(gen.next().value).toStrictEqual(select(getAnalyticsState));
            expect(gen.next(analyticsTestState).value).toStrictEqual(put(setCardSearch(cardSearchUndefinedAuthors)));
            expect(gen.next().done).toBe(true);
        });

        it('should call proper sagas with clean authors and categories', () => {
            const gen: SagaIterator = setFilters(cardSearch);
            expect(gen.next().value).toStrictEqual(select(getAnalyticsState));
            const cleanCardSearch = { name: 'progress B', authorNames: ['author'], categories: ['DevOps COE'], ...defaultPagination };
            expect(gen.next(analyticsTestState).value).toStrictEqual(put(setCardSearch(cleanCardSearch)));
            expect(gen.next().done).toBe(true);
        });
    });

    describe('filterCardsSaga', () => {
        const analyticsTestState = { allCards: cards, favoriteCards };

        it('should call proper sagas when page is 0', () => {
            const cardSearch: CardSearch = {
                ...defaultPagination,
                page: 0,
            };
            const gen: SagaIterator = filterCardsSaga({ payload: cardSearch, type: 'test' });
            expect(gen.next().value).toStrictEqual(select(getAnalyticsState));
            expect(gen.next(analyticsTestState).value).toStrictEqual(call(setFilters, cardSearch));
            expect(gen.next().value).toStrictEqual(put(setIsLoading(true)));
            expect(gen.next().value).toStrictEqual(call(loadAnalyticsCardsForBothTabs));
            expect(gen.next().value).toStrictEqual(put(setIsLoading(false)));
            expect(gen.next().done).toBe(true);
        });

        it('should call proper sagas when page is > 0 and is favorite', () => {
            const cardSearch: CardSearch = {
                ...defaultPagination,
                page: 2,
                isFavorite: true,
            };
            const newFavoriteCards: AnalyticsCardData[] = [
                {
                    contentText: 'New favorite card',
                    id: 'fav-card-2',
                    isFavorite: true,
                },
            ] as never;
            const gen: SagaIterator = filterCardsSaga({ payload: cardSearch, type: 'test' });
            expect(gen.next().value).toStrictEqual(select(getAnalyticsState));
            expect(gen.next(analyticsTestState).value).toStrictEqual(call(setFilters, cardSearch));
            expect(gen.next().value).toStrictEqual(put(setIsLoading(true)));
            expect(gen.next().value).toStrictEqual(call(loadAnalyticsCardsData, true));
            expect(gen.next([newFavoriteCards, 2]).value).toStrictEqual(put(setFavoriteCards(favoriteCards.concat(newFavoriteCards))));
            expect(gen.next(newFavoriteCards).value).toStrictEqual(put(setTotalAvailableFavoriteCards(2)));
            expect(gen.next().value).toStrictEqual(put(setIsLoading(false)));
            expect(gen.next().done).toBe(true);
        });

        it('should call proper sagas when page is > 0 and is NOT favorite', () => {
            const cardSearch: CardSearch = {
                ...defaultPagination,
                page: 2,
                isFavorite: false,
            };
            const newCards: AnalyticsCardData[] = [
                {
                    contentText: 'New card',
                    id: 'non-fav-card-2',
                    isFavorite: false,
                },
            ] as never;
            const gen: SagaIterator = filterCardsSaga({ payload: cardSearch, type: 'non-favorite' });
            expect(gen.next().value).toStrictEqual(select(getAnalyticsState));
            expect(gen.next(analyticsTestState).value).toStrictEqual(call(setFilters, cardSearch));
            expect(gen.next().value).toStrictEqual(put(setIsLoading(true)));
            expect(gen.next().value).toStrictEqual(call(loadAnalyticsCardsData, false));
            expect(gen.next([newCards, 3]).value).toStrictEqual(put(setAllCards(cards.concat(newCards))));
            expect(gen.next(newCards).value).toStrictEqual(put(setTotalAvailableCards(3)));
            expect(gen.next().value).toStrictEqual(put(setIsLoading(false)));
            expect(gen.next().done).toBe(true);
        });
    });

    describe('setCardSearchSaga', () => {
        const setAnalyticsListFilters = jest.fn();
        const update = jest.fn();
        const FiltersQueryParams = {
            update,
        };
        const ClientSettings = {
            setAnalyticsListFilters,
        };

        const runSaga = (payload: CardSearch) => {
            const gen: SagaIterator = setCardSearchSaga({ payload, type: 'test' });
            expect(gen.next().value).toStrictEqual(call(getAngularService, 'FiltersQueryParams'));
            expect(gen.next(FiltersQueryParams).value).toStrictEqual(call(getAngularService, 'ClientSettings'));
            expect(gen.next(ClientSettings).done).toBe(true);
        };

        it('should update the query params and local storage when saving the filters', () => {
            runSaga({ name: 'progress B', ...defaultPagination });
            expect(update).toHaveBeenCalledWith({ authorNames: undefined, categories: undefined, name: 'progress B' });
            expect(setAnalyticsListFilters).toHaveBeenCalledWith({ authorNames: undefined, categories: undefined, name: 'progress B', ...defaultPagination });
        });

        it('should convert categories and authors as csv', () => {
            runSaga({ name: 'progress B', categories: ['cat1', 'cat2'], authorNames: ['author1'], ...defaultPagination });
            expect(update).toHaveBeenCalledWith({ authorNames: 'author1', categories: 'cat1,cat2', name: 'progress B' });
            expect(setAnalyticsListFilters).toHaveBeenCalledWith({
                authorNames: ['author1'],
                categories: ['cat1', 'cat2'],
                name: 'progress B',
                ...defaultPagination,
            });
        });
    });

    describe('loadDashboardSaga', () => {
        const intelligenceConfiguration = {
            id: '1',
            folderColumnName: 'test',
        };
        it('should select the dashboard if found', () => {
            const gen: SagaIterator = loadDashboardSaga({ payload: '1', type: '' });
            expect(gen.next().value).toStrictEqual(call(checkIfIntelligenceIsConfigured));
            expect(gen.next().value).toStrictEqual(select(getAnalyticsState));
            expect(
                gen.next({
                    allCards: [{ id: '1' }],
                    intelligenceConfiguration,
                }).value,
            ).toStrictEqual(put(setSelectedDashboard({ id: '1' } as never)));
            expect(gen.next().done).toBe(true);
        });
        it('should show toastr of not found if the dashboard is not found', () => {
            const gen: SagaIterator = loadDashboardSaga({ payload: '1', type: '' });
            expect(gen.next().value).toStrictEqual(call(checkIfIntelligenceIsConfigured));
            expect(gen.next().value).toStrictEqual(select(getAnalyticsState));
            expect(
                gen.next({
                    allCards: [{ id: '2' }],
                    intelligenceConfiguration,
                }).value,
            ).toStrictEqual(call(toaster.error, "Dashboard [1] can't be found"));
            expect(gen.next().done).toBe(true);
        });
        it('should reload cards', () => {
            const gen: SagaIterator = loadDashboardSaga({ payload: '1', type: '' });
            expect(gen.next().value).toStrictEqual(call(checkIfIntelligenceIsConfigured));
            expect(gen.next().value).toStrictEqual(select(getAnalyticsState));
            expect(
                gen.next({
                    allCards: [],
                    intelligenceConfiguration,
                }).value,
            ).toStrictEqual(call(doPostWithToken, 'api/extension/intelligence/dashboard?serverId=1&dashboardId=1'));

            expect(gen.next({ entity: response.entity.results[0] }).value).toStrictEqual(
                put(
                    setSelectedDashboard({
                        contentText: undefined,
                        embedConfig: undefined,
                        embeddingId: undefined,
                        externalId: 1,
                        folderFilter: undefined,
                        id: 1,
                        imageSrc: 'thumbnail',
                        isFavorite: undefined,
                        originalDashboard: {
                            author_fullname: 'author',
                            categories: ['DevOps COE'],
                            external_id: '1',
                            filter_configuration: [
                                {
                                    key: 'W8E2C0BF756EF4EBC9F04607BC5285770',
                                    name: 'Tags',
                                    source: {
                                        id: '941166EDD92F4BDDBB264D9DEDFE9D2E',
                                        type: 12,
                                    },
                                    summary: 'Tags In List (All)',
                                },
                                {
                                    key: 'W1C5B072BE1174884905C60DDEC2DEB79',
                                    name: 'Folder',
                                    source: {
                                        id: 'EFC67C796B6949D0BD02630332B0A98D',
                                        type: 12,
                                    },
                                    summary: '',
                                },
                                {
                                    key: 'W6B8B4F8CCB1D42DDA736A8BF406CC40D',
                                    name: 'User',
                                    source: {
                                        id: '7C80778BC1B5484388B645DD7F6F2FAB',
                                        type: 12,
                                    },
                                    summary: '',
                                },
                                {
                                    key: 'WF4E7E5C872D34C79B9DD745FB09416BF',
                                    name: 'Start date',
                                    source: {
                                        id: '3D1555170D1542DF9D733E047317ACB3',
                                        type: 12,
                                    },
                                    summary: "(([Workflow start date]@ID Between '4/6/2024' And '5/6/2024 11:59:59 PM'))",
                                },
                                {
                                    key: 'WBA087DF0CB2144D88EDEF312676DFAE8',
                                    name: 'Team',
                                    source: {
                                        id: '8DB0515AC64246709B0BE907393611AB',
                                        type: 12,
                                    },
                                    summary: '',
                                },
                                {
                                    key: 'W699239EE571249E88F678D827B5489E0',
                                    name: 'Workflow category',
                                    source: {
                                        id: '25BC07D902C94923ADC6FA55A4F1A569',
                                        type: 12,
                                    },
                                    summary: 'Workflow category In List (All)',
                                },
                            ],
                            id: 1,
                            thumbnail: 'thumbnail',
                        },
                        serverUrl: undefined,
                        subtitle: 'author',
                        tags: ['DevOps COE'],
                        title: undefined,
                    } as never),
                ),
            );
            expect(gen.next().done).toBe(true);
        });

        it('should not not do anything if intelligenceConfiguration is not set', () => {
            const gen: SagaIterator = loadDashboardSaga({ payload: '1', type: '' });
            expect(gen.next().value).toStrictEqual(call(checkIfIntelligenceIsConfigured));
            expect(gen.next().value).toStrictEqual(select(getAnalyticsState));
            expect(gen.next({ intelligenceConfiguration: undefined }).done).toBe(true);
        });
    });

    describe('getToken', () => {
        it('should call proper sagas when there is no token in reducer and return a token', () => {
            const gen: SagaIterator = getToken();
            expect(gen.next().value).toStrictEqual(select(getAnalyticsState));
            expect(gen.next({ tokenExpiresAt: undefined, tokenValue: undefined }).value).toStrictEqual(call(httpGET, 'oidc/token', true));
            expect(gen.next({ data: { tokenValue: 'mytoken', expiresAt: 1 } }).value).toStrictEqual(
                put(setTokenData({ tokenExpiresAt: 1, tokenValue: 'mytoken' })),
            );
            expect(gen.next()).toStrictEqual({ done: true, value: 'mytoken' });
        });
        it('should call proper sagas when there is a expired token in reducer and return a token', () => {
            const gen: SagaIterator = getToken();
            expect(gen.next().value).toStrictEqual(select(getAnalyticsState));
            expect(gen.next({ tokenExpiresAt: 1, tokenValue: 'mytoken' }).value).toStrictEqual(call(httpGET, 'oidc/token', true));
            expect(gen.next({ data: { tokenValue: 'newToken', expiresAt: 2 } }).value).toStrictEqual(
                put(setTokenData({ tokenExpiresAt: 2, tokenValue: 'newToken' })),
            );
            expect(gen.next()).toStrictEqual({ done: true, value: 'newToken' });
        });
        it('should call proper sagas when there is a not expired token in reducer and return a token', () => {
            const gen: SagaIterator = getToken();
            expect(gen.next().value).toStrictEqual(select(getAnalyticsState));
            // year 2099
            expect(gen.next({ tokenExpiresAt: 4082085612, tokenValue: 'mytoken' })).toStrictEqual({ done: true, value: 'mytoken' });
        });
        it('should show toastr in case of error', () => {
            try {
                const gen: SagaIterator = getToken();
                expect(gen.next().value).toStrictEqual(select(getAnalyticsState));
                expect(gen.next({ tokenExpiresAt: undefined, tokenValue: undefined }).value).toStrictEqual(call(httpGET, 'oidc/token', true));
                expect(gen.throw?.('error').value).toStrictEqual(
                    call(toaster.error, 'Something went wrong while connecting to Analytics. Your session might be expired.'),
                );
                expect(gen.next().done).toBe(true);
            } catch (e) {
                expect(e).toBe('error');
            }
        });
    });

    describe('doPostWithToken', () => {
        it('should call proper sagas and return the data response', () => {
            const gen: SagaIterator = doPostWithToken('url');
            expect(gen.next().value).toStrictEqual(call(getToken));
            expect(gen.next('mytoken').value).toStrictEqual(call(httpPOST, 'url', { accessToken: 'mytoken' }));
            expect(gen.next({ data: 'response' })).toStrictEqual({ done: true, value: 'response' });
        });
    });
});
