import {
    calculateLastLoadedPage,
    calculateNumberOfFiltersApplied,
    calculateTitleAndSubtitleForEmptyState,
    convertArrayToString,
    createAnalyticsApiUrl,
    extractRegionFromUrl,
    getAuthorName,
    getFavoriteIcon,
    getFavoriteText,
    getFilteredSampleCards,
    getFolderFilter,
    getUpdatedCards,
    mapAuthorName,
    mapSupersetToAnalyticsCards,
    removeAuthorFromFilters,
    removeCategoryFromFilters,
} from './index';
import { CardSearch } from '../ducks/analytics.reducer';
import { AnalyticsCardData, IntelligenceHttpConnection, DashboardView } from '../types';
import { ANALYTICS_AUTHORS, ANALYTICS_CATEGORIES, ANALYTICS_SAMPLE_CARDS, DIGITAL_AI_AUTHOR } from '../constants';

describe('analytics helpers', () => {
    describe('calculateNumberOfFiltersApplied', () => {
        it('should return 0 when cardSearch is undefined', () => {
            const result = calculateNumberOfFiltersApplied(undefined);
            expect(result).toBe(0);
        });

        it('should return 0 when both categories and authorNames are empty arrays and no name', () => {
            const cardSearch: CardSearch = {
                categories: [],
                authorNames: [],
                name: '',
            } as never;
            const result = calculateNumberOfFiltersApplied(cardSearch);
            expect(result).toBe(0);
        });

        it('should return 1 when name is present', () => {
            const cardSearch: CardSearch = {
                categories: [],
                authorNames: [],
                name: 'my name',
            } as never;
            const result = calculateNumberOfFiltersApplied(cardSearch);
            expect(result).toBe(1);
        });

        it('should return the total number of filters applied when categories and authorNames have elements', () => {
            const cardSearch: CardSearch = {
                categories: ['Category 1'],
                authorNames: ['Author 1', 'Author 2'],
            } as never;
            const result = calculateNumberOfFiltersApplied(cardSearch);
            expect(result).toBe(3);
        });

        it('should return the number of filters applied based on the provided categories and authorNames', () => {
            const cardSearch: CardSearch = {
                categories: ['Category 1', 'Category 2'],
                authorNames: ['Author 1'],
            } as never;
            const result = calculateNumberOfFiltersApplied(cardSearch);
            expect(result).toBe(3);
        });
    });

    describe('removeCategoryFromFilters', () => {
        it('should return undefined when filteredCategories is undefined', () => {
            const result = removeCategoryFromFilters('Category 1', undefined);
            expect(result).toBeUndefined();
        });

        it('should return an empty array when filteredCategories is an empty array', () => {
            const result = removeCategoryFromFilters('Category 1', []);
            expect(result).toEqual([]);
        });

        it('should remove the specified category from the filteredCategories', () => {
            const category1 = 'Category 1';
            const category2 = 'Category 2';
            const category3 = 'Category 3';

            const filteredCategories: string[] = [category1, category2, category3];
            const result = removeCategoryFromFilters('Category 2', filteredCategories);
            expect(result).toEqual([category1, category3]);
        });

        it('should not modify the original filteredCategories array', () => {
            const category1 = 'Category 1';
            const category2 = 'Category 2';
            const category3 = 'Category 3';

            const filteredCategories: string[] = [category1, category2, category3];
            removeCategoryFromFilters('Category 2', filteredCategories);
            expect(filteredCategories).toEqual([category1, category2, category3]);
        });
    });

    describe('removeAuthorFromFilters', () => {
        it('should return undefined when filteredAuthorNames is undefined', () => {
            const result = removeAuthorFromFilters('Author 1', undefined);
            expect(result).toBeUndefined();
        });

        it('should return an empty array when filteredAuthorNames is an empty array', () => {
            const result = removeAuthorFromFilters('Author 1', []);
            expect(result).toEqual([]);
        });

        it('should remove the specified authorName from the filteredAuthorNames', () => {
            const authorNames = ['Author 1', 'Author 2', 'Author 3'];
            const result = removeAuthorFromFilters('Author 2', authorNames);
            expect(result).toEqual(['Author 1', 'Author 3']);
        });

        it('should not modify the original filteredAuthorNames array', () => {
            const authorNames = ['Author 1', 'Author 2', 'Author 3'];
            removeAuthorFromFilters('Author 2', authorNames);
            expect(authorNames).toEqual(['Author 1', 'Author 2', 'Author 3']);
        });
    });

    describe('convertArrayToString', () => {
        it('should return undefined if undefined', () => {
            expect(convertArrayToString(undefined)).toBeUndefined();
        });
        it('should return undefined if empty', () => {
            expect(convertArrayToString([])).toBeUndefined();
        });
        it('should return csv if there is something', () => {
            expect(convertArrayToString(['a', 'b'])).toBe('a,b');
        });
    });

    describe('calculateTitleAndSubtitleForEmptyState', () => {
        it('should return correct title and subtitle when on favorite tab', () => {
            const favoriteTabData = {
                subtitle: 'To easily add dashboards to your favorites, simply click the star icon located next to each of the dashboards. 🌟',
                title: 'No favorites have been added so far',
            };
            expect(calculateTitleAndSubtitleForEmptyState(true, true)).toStrictEqual(favoriteTabData);
            expect(calculateTitleAndSubtitleForEmptyState(true, false)).toStrictEqual(favoriteTabData);
        });
        it('should return correct title and subtitle when not on favorite tab, depending if analytics connection is configured', () => {
            expect(calculateTitleAndSubtitleForEmptyState(false, true)).toStrictEqual({
                subtitle: 'Create and manage Analytics dashboards',
                title: 'Analytics dashboards',
            });
            expect(calculateTitleAndSubtitleForEmptyState(false, false)).toStrictEqual({
                subtitle: 'There are no available dashboards yet.',
                title: 'No dashboards yet',
            });
        });
    });

    describe('getFolderFilter', () => {
        const mockDashboard: DashboardView = {
            application_instances: [],
            author_id: '',
            bi_type: DashboardView.bi_type.MICROSTRATEGY,
            dashboard_url: '',
            external_object: {},
            help_content_id: '',
            target_apps: ['RELEASE'],
            author_fullname: '',
            thumbnail: '',
            categories: [],
            created_dt: '',
            embed_config: {} as never,
            description: '',
            external_embedding_id: '',
            external_id: '',
            favorite: false,
            filter_configuration: [
                {
                    summary: 'Tags In List (All)',
                    name: 'Tags',
                    source: {
                        id: '941166EDD92F4BDDBB264D9DEDFE9D2E',
                        type: 12,
                    },
                    key: 'W8E2C0BF756EF4EBC9F04607BC5285770',
                },
                {
                    summary: '',
                    name: 'Folder',
                    source: {
                        id: 'EFC67C796B6949D0BD02630332B0A98D',
                        type: 12,
                    },
                    key: 'W1C5B072BE1174884905C60DDEC2DEB79',
                },
                {
                    summary: '',
                    name: 'User',
                    source: {
                        id: '7C80778BC1B5484388B645DD7F6F2FAB',
                        type: 12,
                    },
                    key: 'W6B8B4F8CCB1D42DDA736A8BF406CC40D',
                },
                {
                    summary: "(([Workflow start date]@ID Between '4/6/2024' And '5/6/2024 11:59:59 PM'))",
                    name: 'Start date',
                    source: {
                        id: '3D1555170D1542DF9D733E047317ACB3',
                        type: 12,
                    },
                    key: 'WF4E7E5C872D34C79B9DD745FB09416BF',
                },
                {
                    summary: '',
                    name: 'Team',
                    source: {
                        id: '8DB0515AC64246709B0BE907393611AB',
                        type: 12,
                    },
                    key: 'WBA087DF0CB2144D88EDEF312676DFAE8',
                },
                {
                    summary: 'Workflow category In List (All)',
                    name: 'Workflow category',
                    source: {
                        id: '25BC07D902C94923ADC6FA55A4F1A569',
                        type: 12,
                    },
                    key: 'W699239EE571249E88F678D827B5489E0',
                },
            ],
            id: '',
            is_ootb_dashboard: true,
            lifecycle_state: DashboardView.lifecycle_state.DRAFT,
            name: '',
            server_url: '',
        };

        it('should return config if folderColumn is in native filter configuration', () => {
            const result = getFolderFilter(mockDashboard, 'Folder');
            expect(result).toStrictEqual({
                key: 'W1C5B072BE1174884905C60DDEC2DEB79',
                name: 'Folder',
                source: {
                    id: 'EFC67C796B6949D0BD02630332B0A98D',
                    type: 12,
                },
                summary: '',
            });
        });

        it('should return undefined if folderColumn is not in native filter configuration', () => {
            const result = getFolderFilter(mockDashboard, 'otherColumn');
            expect(result).toBeUndefined();
        });

        it('should return undefined if there is no filter configuration', () => {
            const result = getFolderFilter({ ...mockDashboard, filter_configuration: [] }, 'Folder');
            expect(result).toBeUndefined();
        });
    });

    describe('getAuthorName', () => {
        const dashboardItem: DashboardView = {
            application_instances: [],
            author_id: '',
            bi_type: DashboardView.bi_type.MICROSTRATEGY,
            dashboard_url: '',
            external_object: {},
            help_content_id: '',
            target_apps: ['RELEASE'],
            author_fullname: 'author',
            categories: ['DevOps COE'],
            created_dt: 'created_dt',
            embed_config: {} as never,
            description: 'description',
            external_embedding_id: 'external_embedding_id',
            external_id: '14',
            favorite: false,
            filter_configuration: [],
            id: '14',
            is_ootb_dashboard: true,
            lifecycle_state: DashboardView.lifecycle_state.PUBLISHED,
            name: 'name',
            server_url: 'api/v1/dashboard/14',
            thumbnail: 'thumbnailBase64',
        };

        it('should return "Digital.ai" as author if is_ootb_dashboard is true', () => {
            expect(getAuthorName(dashboardItem)).toStrictEqual(DIGITAL_AI_AUTHOR);
        });

        it('should return author_fullname value as author if is_ootb_dashboard is false', () => {
            expect(getAuthorName({ ...dashboardItem, is_ootb_dashboard: false })).toStrictEqual('author');
        });
    });

    describe('mapSupersetToAnalyticsCards', () => {
        const response = [
            {
                id: '14',
                external_id: '14',
                name: 'name',
                created_dt: 'created_dt',
                description: 'description',
                favorite: false,
                thumbnail: 'thumbnailBase64',
                external_embedding_id: 'external_embedding_id',
                server_url: 'api/v1/dashboard/14',
                embed_config: {},
            },
        ] as Array<DashboardView>;
        it('should map as much as possible', () => {
            expect(mapSupersetToAnalyticsCards(response, '')).toStrictEqual([
                {
                    contentText: 'description',
                    embeddingId: 'external_embedding_id',
                    externalId: 14,
                    folderFilter: undefined,
                    id: '14',
                    imageSrc: 'thumbnailBase64',
                    isFavorite: false,
                    originalDashboard: {
                        created_dt: 'created_dt',
                        description: 'description',
                        embed_config: {},
                        external_embedding_id: 'external_embedding_id',
                        external_id: '14',
                        favorite: false,
                        id: '14',
                        name: 'name',
                        server_url: 'api/v1/dashboard/14',
                        thumbnail: 'thumbnailBase64',
                    },
                    serverUrl: 'api/v1/dashboard/14',
                    title: 'name',
                    subtitle: undefined,
                    tags: undefined,
                    embedConfig: {},
                },
            ]);
        });
    });

    describe('mapAuthorName', () => {
        const response = [
            {
                id: 'id1',
                full_name: ' ',
            },
            {
                id: 'id2',
                full_name: 'author 1',
            },
            {
                id: 'id3',
                full_name: 'author 2',
            },
        ] as never;
        it('should map authors to string', () => {
            expect(mapAuthorName(response)).toStrictEqual([DIGITAL_AI_AUTHOR, 'author 1', 'author 2']);
        });
    });

    describe('getFavoriteIcon', () => {
        it('should return icon', () => {
            expect(getFavoriteIcon(true)).toBe('star-favorites-active');
            expect(getFavoriteIcon(false)).toBe('star-favorites-default');
        });
    });

    describe('getFavoriteText', () => {
        it('should return button text', () => {
            expect(getFavoriteText(true)).toBe('Remove from favorites');
            expect(getFavoriteText(false)).toBe('Add to favorites');
        });
    });

    describe('createAnalyticsApiUrl', () => {
        const intelligenceConfiguration: IntelligenceHttpConnection = {
            id: '123',
            folderColumnName: 'Test Folder',
        } as never;

        it('should create a URL with pagination only', () => {
            const cardSearch = {
                itemsPerPage: 10,
                page: 1,
                isFavorite: false,
            };

            const result = createAnalyticsApiUrl(cardSearch, intelligenceConfiguration.id);
            expect(result).toBe('api/extension/intelligence/dashboards?serverId=123&page=1&resultsPerPage=10');
        });

        it('should include name filter if provided', () => {
            const cardSearch = {
                itemsPerPage: 10,
                page: 1,
                name: 'Test Dashboard',
                isFavorite: false,
            };

            const result = createAnalyticsApiUrl(cardSearch, intelligenceConfiguration.id);
            expect(result).toContain('name[isearch]:Test%20Dashboard');
        });

        it('should include categories filter if provided', () => {
            const cardSearch = {
                itemsPerPage: 10,
                page: 1,
                categories: ['Category1', 'Category2'],
                isFavorite: false,
            };

            const result = createAnalyticsApiUrl(cardSearch, intelligenceConfiguration.id);
            expect(result).toContain('categories[in]:Category1%2CCategory2');
        });

        it('should include author names filter', () => {
            const cardSearch = {
                itemsPerPage: 10,
                page: 1,
                authorNames: ['John Doe', 'Jane Doe'],
                isFavorite: false,
            };

            const result = createAnalyticsApiUrl(cardSearch, intelligenceConfiguration.id);
            expect(result).toContain('author_fullname[in]:John%20Doe%2CJane%20Doe');
        });

        it('should handle "Digital.ai" author with OOTB filter', () => {
            const cardSearch = {
                itemsPerPage: 10,
                page: 1,
                authorNames: [DIGITAL_AI_AUTHOR, 'John Doe'],
                isFavorite: false,
            };

            const result = createAnalyticsApiUrl(cardSearch, intelligenceConfiguration.id);

            expect(result).toContain('is_ootb_dashboard:1');
            expect(result).toContain('author_fullname[in]:John%20Doe');
        });

        it('should create URL with multiple filters', () => {
            const cardSearch = {
                itemsPerPage: 10,
                page: 1,
                name: 'Test',
                categories: ['Category1'],
                authorNames: ['John Doe'],
                isFavorite: false,
            };

            const result = createAnalyticsApiUrl(cardSearch, intelligenceConfiguration.id);

            expect(result).toContain('name[isearch]:Test');
            expect(result).toContain('categories[in]:Category1');
            expect(result).toContain('author_fullname[in]:John%20Doe');
        });

        it('should create URL with orFilters', () => {
            const cardSearch = {
                itemsPerPage: 10,
                page: 1,
                name: 'Test',
                categories: ['Category1'],
                authorNames: [DIGITAL_AI_AUTHOR, 'John Doe'],
                isFavorite: false,
            };

            const result = createAnalyticsApiUrl(cardSearch, intelligenceConfiguration.id);

            const filters = result.split('&filter=').slice(1);
            expect(filters.length).toBe(2);

            expect(filters[0]).toContain('is_ootb_dashboard:1');
            expect(filters[1]).toContain('author_fullname[in]:John%20Doe');
        });

        it('should create URL for favorites filtering', () => {
            const cardSearch = {
                itemsPerPage: 10,
                page: 1,
                isFavorite: true,
            };

            const result = createAnalyticsApiUrl(cardSearch, intelligenceConfiguration.id);
            expect(result).toContain('&favorite=true');
        });
    });

    describe('extractRegionFromUrl', () => {
        it('should parse the url and get the region', () => {
            expect(extractRegionFromUrl('https://www.google.com')).toBeUndefined();
            expect(extractRegionFromUrl('https://www.digital.ai')).toBeUndefined();
            expect(extractRegionFromUrl('https://digital.ai')).toBeUndefined();
            expect(extractRegionFromUrl('https://vanity.digital.ai')).toBeUndefined();
            expect(extractRegionFromUrl('https://vanity.region.digital.ai')).toBe('region');
        });
    });

    describe('getUpdatedCards', () => {
        const cards: AnalyticsCardData[] = [
            {
                id: '1',
                isFavorite: false,
            },
            {
                id: '2',
                isFavorite: true,
            },
        ] as never;

        it('should update the card if found', () => {
            const updatedCard: AnalyticsCardData = {
                id: '1',
                isFavorite: true,
            } as never;
            const result = getUpdatedCards(updatedCard, cards);
            expect(result).toEqual([
                {
                    ...cards[0],
                    ...updatedCard,
                },
                cards[1],
            ]);
        });

        it('should return the original array if card is not found', () => {
            const updatedCard: AnalyticsCardData = {
                id: '3', // non-existent ID
                isFavorite: true,
            } as never;
            const result = getUpdatedCards(updatedCard, cards);
            expect(result).toEqual(cards);
        });
    });

    describe('calculateLastLoadedPage', () => {
        it('should return 0 when numberOfCards is 0', () => {
            expect(calculateLastLoadedPage(0, 10)).toBe(0);
        });

        it('should return 0 when numberOfCards is less than itemsPerPage', () => {
            expect(calculateLastLoadedPage(5, 10)).toBe(0);
        });

        it('should return 0 when numberOfCards is equal to itemsPerPage', () => {
            expect(calculateLastLoadedPage(10, 10)).toBe(0);
        });

        it('should return 1 when numberOfCards is more than itemsPerPage but less than 2 * itemsPerPage', () => {
            expect(calculateLastLoadedPage(15, 10)).toBe(1);
        });

        it('should return correct page number when numberOfCards is multiple of itemsPerPage', () => {
            expect(calculateLastLoadedPage(20, 10)).toBe(1);
            expect(calculateLastLoadedPage(30, 10)).toBe(2);
        });

        it('should return correct page number when numberOfCards is not a multiple of itemsPerPage', () => {
            expect(calculateLastLoadedPage(25, 10)).toBe(2);
            expect(calculateLastLoadedPage(35, 10)).toBe(3);
        });
    });

    describe('getFilteredSampleCards', () => {
        it('should return all cards if no search criteria is provided', () => {
            const result = getFilteredSampleCards();
            expect(result).toEqual(ANALYTICS_SAMPLE_CARDS);
        });

        it('should filter cards by name', () => {
            const cardSearch: CardSearch = { name: 'onboarding' } as never;
            const result = getFilteredSampleCards(cardSearch);
            expect(result).toEqual([ANALYTICS_SAMPLE_CARDS[0]]);
        });

        it('should filter cards by author names', () => {
            const cardSearch: CardSearch = { authorNames: [ANALYTICS_AUTHORS[0]] } as never;
            const result = getFilteredSampleCards(cardSearch);
            expect(result).toEqual(ANALYTICS_SAMPLE_CARDS);
        });

        it('should filter cards by categories', () => {
            const cardSearch: CardSearch = { categories: [ANALYTICS_CATEGORIES[1]] } as never;
            const result = getFilteredSampleCards(cardSearch);
            expect(result).toEqual([ANALYTICS_SAMPLE_CARDS[1]]);
        });

        it('should filter cards by name and author names', () => {
            const cardSearch: CardSearch = { name: 'dora', authorNames: [ANALYTICS_AUTHORS[0]] } as never;
            const result = getFilteredSampleCards(cardSearch);
            expect(result).toEqual([ANALYTICS_SAMPLE_CARDS[1]]);
        });

        it('should filter cards by name and categories', () => {
            const cardSearch: CardSearch = { name: 'release', categories: [ANALYTICS_CATEGORIES[4]] } as never;
            const result = getFilteredSampleCards(cardSearch);
            expect(result).toEqual([ANALYTICS_SAMPLE_CARDS[2], ANALYTICS_SAMPLE_CARDS[3], ANALYTICS_SAMPLE_CARDS[4], ANALYTICS_SAMPLE_CARDS[5]]);
        });

        it('should filter cards by author names and categories', () => {
            const cardSearch: CardSearch = { authorNames: [ANALYTICS_AUTHORS[0]], categories: [ANALYTICS_CATEGORIES[4]] } as never;
            const result = getFilteredSampleCards(cardSearch);
            expect(result).toEqual([ANALYTICS_SAMPLE_CARDS[2], ANALYTICS_SAMPLE_CARDS[3], ANALYTICS_SAMPLE_CARDS[4], ANALYTICS_SAMPLE_CARDS[5]]);
        });

        it('should filter cards by all criteria', () => {
            const cardSearch: CardSearch = { name: 'release', authorNames: [ANALYTICS_AUTHORS[0]], categories: [ANALYTICS_CATEGORIES[4]] } as never;
            const result = getFilteredSampleCards(cardSearch);
            expect(result).toEqual([ANALYTICS_SAMPLE_CARDS[2], ANALYTICS_SAMPLE_CARDS[3], ANALYTICS_SAMPLE_CARDS[4], ANALYTICS_SAMPLE_CARDS[5]]);
        });

        it('should return no cards if no match is found', () => {
            const cardSearch: CardSearch = { name: 'nonexistent' } as never;
            const result = getFilteredSampleCards(cardSearch);
            expect(result).toEqual([]);
        });
    });
});
