import React from 'react';
import { DotButton, DotCard, DotCardContent, DotCardFooter, DotCardHeader, DotTypography } from '@digital-ai/dot-components';
import { AnalyticsCard, AnalyticsCardProps } from './analytics-card.component';
import { mockResizeObserver, mountWithTheme, ReactWrapper } from '@xlr-ui/tests/unit/testing-utils';
import { ChipGroup } from '@xlr-ui/app/react/components/chip-group/chip-group.component';
import { AnalyticsFavoriteIconButton } from './dashboard/analytics-favorite-icon-button.component';

describe('AnalyticsCard', () => {
    const id = '655285cdd75b4c64fc5bead2';
    const externalId = 123;
    const embeddingId = '8dd03d24-008c-450c-9bbe-f7a21e2a5033';
    const serverUrl = 'https://intelligence-devops.bi.staging.digital.ai';

    const contentText = 'My content text';
    const imageSrc = 'image-src';
    const isFavorite = false;
    const isUpdating = false;
    const onFavoriteClick = jest.fn();
    const onOpenDashboardClick = jest.fn();
    const subtitle = 'My subtitle';
    const tags = ['tag1', 'tag2'];
    const title = 'My card title';
    const isAnalyticsConfigured = true;

    let wrapper: ReactWrapper;

    const defaultProps: AnalyticsCardProps = {
        embeddingId,
        externalId,
        contentText,
        id,
        imageSrc,
        embedConfig: {} as never,
        isAnalyticsConfigured,
        isFavorite,
        isUpdating,
        onFavoriteClick,
        onOpenDashboardClick,
        serverUrl,
        subtitle,
        tags,
        title,
    };

    const mount = (props: AnalyticsCardProps = defaultProps) => {
        wrapper = mountWithTheme(<AnalyticsCard {...props} />);
    };

    beforeAll(() => {
        mockResizeObserver();
    });

    afterAll(() => {
        jest.restoreAllMocks();
    });

    afterEach(() => {
        jest.resetAllMocks();
    });

    const getDotCard = () => wrapper.find(DotCard);
    const getAnalyticsFavoriteIconButton = () => wrapper.findWhere((node) => node.is(AnalyticsFavoriteIconButton));
    const getDotCardHeader = () => getDotCard().find(DotCardHeader);
    const getDotCardContent = () => getDotCard().find(DotCardContent);
    const getImage = () => getDotCard().find('img');
    const getContentTypography = () => getDotCardContent().findWhere((node) => node.is(DotTypography) && node.props()['data-testid'] === 'content-typography');
    const getChipGroup = () => getDotCardContent().find(ChipGroup);
    const getDotCardFooter = () => getDotCard().find(DotCardFooter);
    const getOpenDashboardButton = () => getDotCardFooter().findWhere((node) => node.is(DotButton) && node.props().className === 'open-dashboard-btn');

    describe('default render', () => {
        it('should render with the default props', () => {
            mount();
            expect(getDotCard()).toExist();

            const dotCardHeader = getDotCardHeader();
            expect(dotCardHeader).toExist();
            expect(dotCardHeader.props().className).toBe('card-header');
            expect(dotCardHeader.props().subheader).toBe(`by ${subtitle}`);
            expect(dotCardHeader.props().title).toBe(title);
            expect(dotCardHeader.props().titleSize).toBe('medium');

            const dotCardContent = getDotCardContent();
            expect(dotCardContent).toExist();
            expect(dotCardContent.props().className).toBe('card-content');

            const analyticsFavoriteIconButton = getAnalyticsFavoriteIconButton();
            expect(analyticsFavoriteIconButton).toExist();
            const analyticsFavoriteIconButtonProps = analyticsFavoriteIconButton.props();
            expect(analyticsFavoriteIconButtonProps.isAnalyticsConfigured).toBe(isAnalyticsConfigured);
            expect(analyticsFavoriteIconButtonProps.isFavorite).toBe(isFavorite);
            expect(analyticsFavoriteIconButtonProps.isUpdating).toBe(isUpdating);
            expect(analyticsFavoriteIconButtonProps.onFavoriteClick).toStrictEqual(expect.any(Function));

            const image = getImage();
            expect(image).toExist();
            expect(image.prop('alt')).toBe('Analytics card image');
            expect(image.prop('src')).toBe(`data:image/svg+xml;base64,${imageSrc}`);

            const contentTypography = getContentTypography();
            expect(contentTypography).toExist();
            expect(contentTypography.props().variant).toBe('body1');
            expect(contentTypography.props().children).toBe(contentText);

            const chipGroup = getChipGroup();
            expect(chipGroup).toExist();
            expect(chipGroup.props().labels).toBe(tags);

            const cardFooter = getDotCardFooter();
            expect(cardFooter).toExist();

            const openDashboardButton = getOpenDashboardButton();
            expect(openDashboardButton).toExist();
            expect(openDashboardButton.props().type).toBe('outlined');
        });

        it('should use default subtitle and image if not provided', () => {
            mount({ ...defaultProps, subtitle: '', imageSrc: '' });
            expect(getDotCardHeader().props().subheader).toBe('by Digital.ai');
            expect(getImage().prop('src')).toBe('static/@project.version@/include/analytics-dashboard/assets/custom.svg');
        });

        it('should render AnalyticsFavoriteIconButton component with isFavorite set to true  when main isFavorite is true', () => {
            mount({ ...defaultProps, isFavorite: true });
            expect(getAnalyticsFavoriteIconButton().props().isFavorite).toBe(true);
        });

        it('should trigger correct handler when Open dashboard button is clicked', () => {
            mount();
            const openDashboardBtn = getOpenDashboardButton();
            openDashboardBtn.invoke('onClick')();
            expect(onOpenDashboardClick).toHaveBeenCalledTimes(1);
        });

        it('should render AnalyticsFavoriteIconButton component with isUpdating set to true  when main isUpdating is true', () => {
            mount({ ...defaultProps, isUpdating: true });
            const analyticsFavoriteIconButton = getAnalyticsFavoriteIconButton();
            expect(analyticsFavoriteIconButton).toExist();
            const analyticsFavoriteIconButtonProps = analyticsFavoriteIconButton.props();
            expect(analyticsFavoriteIconButtonProps.isUpdating).toBe(true);
        });
    });

    describe('when analytics is not configured', () => {
        beforeEach(() => {
            mount({ ...defaultProps, isAnalyticsConfigured: false });
        });

        it('should set isAnalyticsConfigured to false on AnalyticsFavoriteIconButton component', () => {
            const analyticsFavoriteIconButton = getAnalyticsFavoriteIconButton();
            expect(analyticsFavoriteIconButton.props().isAnalyticsConfigured).toBe(false);
        });

        it('should add data-testid to open dashboard button', () => {
            const openDashboardButton = getOpenDashboardButton();
            expect(openDashboardButton.props()['data-testid']).toBe('open-analytics-dashboard-btn');
        });
    });
});
