import React from 'react';
import { mountWithTheme, ReactWrapper } from '@xlr-ui/tests/unit/testing-utils';
import { AnalyticsPageFilterDrawer, AnalyticsPageFilterDrawerProps } from './analytics-page-filter-drawer.component';
import { CardSearch } from '../ducks/analytics.reducer';
import { DotAutoComplete, DotCheckbox, DotChip, DotTypography } from '@digital-ai/dot-components';
import { AnalyticsCategoriesSkeleton } from './analytics-categories.skeleton';
import { DEFAULT_PAGINATION } from '@xlr-ui/app/constants/pagination';
import { FilterDrawer } from '@xlr-ui/app/features/common/components/filter-drawer/filter-drawer.component';

describe('AnalyticsPageFilterDrawer', () => {
    let wrapper: ReactWrapper;
    const defaultPagination = DEFAULT_PAGINATION;
    const authors = ['author1', 'author2', 'author3'];
    const categories = ['Cat1', 'Cat2', 'Cat3', 'Cat4'];
    const cardSearch: CardSearch = { categories: [categories[0], categories[1]], authorNames: [authors[0]], ...defaultPagination };
    const onCardFilter = jest.fn();
    const onClearAll = jest.fn();
    const onDrawerClose = jest.fn();
    const numberOfFiltersApplied = 3;

    const defaultProps: AnalyticsPageFilterDrawerProps = {
        authors,
        cardSearch,
        categories,
        isCategoriesLoading: false,
        isDrawerOpened: true,
        numberOfFiltersApplied,
        onCardFilter,
        onClearAll,
        onDrawerClose,
    };

    const getFilterDrawer = () => wrapper.find(FilterDrawer);

    const getFilterGroupTitleDiv = () => wrapper.find('div.filter-group-title');
    const getFilterByCategoryTitleTypography = () => getFilterGroupTitleDiv().at(0).find(DotTypography);

    const getFilterCheckboxGroup = () => wrapper.find('div.filter-checkbox-group');

    const getCategoriesCheckboxes = () => getFilterCheckboxGroup().at(0).find(DotCheckbox);
    const getAnalyticsCategoriesSkeleton = () => wrapper.find(AnalyticsCategoriesSkeleton);

    const getAuthorAutocomplete = () => getFilterCheckboxGroup().at(1).find(DotAutoComplete);
    const getAuthorAutocompleteChips = () => getAuthorAutocomplete().find(DotChip);

    const mount = (props = defaultProps) => {
        wrapper = mountWithTheme(<AnalyticsPageFilterDrawer {...props} />);
    };

    afterEach(() => jest.resetAllMocks());

    it('should render with default props', () => {
        mount();
        const drawer = getFilterDrawer();
        expect(drawer).toExist();
        const drawerProps = drawer.props();
        expect(drawerProps.isOpened).toBe(true);
        expect(drawerProps.numberOfFiltersApplied).toBe(numberOfFiltersApplied);
        expect(drawerProps.onClearAllFilters).toStrictEqual(expect.any(Function));
        expect(drawerProps.onClose).toStrictEqual(expect.any(Function));

        drawer.invoke('onClose')?.();
        expect(onDrawerClose).toHaveBeenCalledTimes(1);

        drawer.invoke('onClearAllFilters')?.();
        expect(onClearAll).toHaveBeenCalledTimes(1);

        expect(getFilterGroupTitleDiv()).toExist();

        const filterByCategoryTitleTypography = getFilterByCategoryTitleTypography();
        expect(filterByCategoryTitleTypography).toExist();
        expect(filterByCategoryTitleTypography.props().variant).toBe('overline');

        expect(getAnalyticsCategoriesSkeleton()).not.toExist();

        expect(getFilterCheckboxGroup()).toExist();
        const categoriesCheckboxes = getCategoriesCheckboxes();
        expect(categoriesCheckboxes.length).toBe(categories.length);

        expect(categoriesCheckboxes.at(0).props().checked).toBe(true);
        expect(categoriesCheckboxes.at(0).props().label).toBe(categories[0]);

        expect(categoriesCheckboxes.at(1).props().checked).toBe(true);
        expect(categoriesCheckboxes.at(1).props().label).toBe(categories[1]);

        expect(categoriesCheckboxes.at(2).props().checked).toBe(false);
        expect(categoriesCheckboxes.at(2).props().label).toBe(categories[2]);

        expect(categoriesCheckboxes.at(3).props().checked).toBe(false);
        expect(categoriesCheckboxes.at(3).props().label).toBe(categories[3]);

        const authorFilterTitle = getFilterCheckboxGroup().at(1);
        expect(authorFilterTitle).toExist();

        const authorAutocomplete = getAuthorAutocomplete().props();

        expect(authorAutocomplete.options).toStrictEqual([{ title: 'author1' }, { title: 'author2' }, { title: 'author3' }]);
        expect(authorAutocomplete.value).toStrictEqual([{ title: 'author1' }]);
        expect(authorAutocomplete.freesolo).toBe(false);
        expect(authorAutocomplete.multiple).toBe(true);

        const authorChips = getAuthorAutocompleteChips();
        expect(authorChips.length).toBe(cardSearch.authorNames?.length);
    });

    it('should execute onCardFilter handler with correct props when category checkbox is checked', () => {
        mount();
        const categoriesCheckboxes = getCategoriesCheckboxes();
        categoriesCheckboxes.at(3).invoke('onChange')?.({ target: { checked: true } } as never, '123');
        expect(onCardFilter).toHaveBeenCalledWith({
            ...cardSearch,
            categories: [categories[0], categories[1], categories[3]],
        });
    });

    it('should execute onCardFilter handler with correct props when category checkbox is unchecked', () => {
        mount();
        const categoriesCheckboxes = getCategoriesCheckboxes();
        categoriesCheckboxes.at(0).invoke('onChange')?.({ target: { checked: false } } as never, '123');
        expect(onCardFilter).toHaveBeenCalledWith({ ...cardSearch, categories: [categories[1]] });
    });

    it('should execute onCardFilter handler with correct props when author is selected', () => {
        mount();
        const authorAutocomplete = getAuthorAutocomplete();
        authorAutocomplete.invoke('onChange')?.('' as never, [{ title: authors[0] }, { title: authors[1] }], '' as never);
        expect(onCardFilter).toHaveBeenCalledWith({ ...cardSearch, authorNames: [authors[0], authors[1]] });
    });

    it('should execute onCardFilter handler with correct props when author is removed from the chip', () => {
        mount();
        const authorChip = getAuthorAutocompleteChips().at(0);
        authorChip.invoke('onDelete')?.('' as never);
        expect(onCardFilter).toHaveBeenCalledWith({ ...cardSearch, authorNames: [] });
    });

    it('should render categories skeleton component when isCategoriesLoading is set to true', () => {
        mount({ ...defaultProps, isCategoriesLoading: true });
        expect(getAnalyticsCategoriesSkeleton()).toExist();
    });
});
