import { SagaIterator } from 'redux-saga';
import { all, call, put, takeEvery, takeLatest } from 'redux-saga/effects';
import { applicationManagement } from './managed-application.reducer';
import {
    managedApplicationSaga,
    executeFetchManagedApplicationsAction,
    executeFetchTemplatesAction,
    getFiltersAction,
    storeFiltersAction,
    createReleaseAction,
} from './managed-application.saga';
import { httpGET, httpPOST } from '../../../../../../../../../core/xlr-ui/app/features/common/services/http';
import { navigation } from '../../../../../../../../../core/xlr-ui/app/features/main-navigation/ducks/navigation.reducer';

import { Template } from '../managed-application.types';

const { navigateTo } = navigation.actions;

const { createRelease, fetchTemplates, getFilters, loadManagedApplications, setTemplates, storeFilters } = applicationManagement.actions;

describe('applicationManagementSaga', () => {
    it('should yield all effects', () => {
        const gen: SagaIterator = managedApplicationSaga();
        expect(gen.next().value).toStrictEqual(
            all([
                takeLatest(fetchTemplates, executeFetchTemplatesAction),
                takeLatest(loadManagedApplications, executeFetchManagedApplicationsAction),
                takeEvery(storeFilters, storeFiltersAction),
                takeEvery(getFilters, getFiltersAction),
                takeLatest(createRelease, createReleaseAction),
            ]),
        );
        expect(gen.next().done).toBe(true);
    });

    describe('executeFetchTemplatesAction', () => {
        const cis = [{ id: 'template-1', title: 'title-1', description: 'description-1' }] as never;
        const applicationPipelineTag = 'the-tag';
        it('should call fetchTemplatesAction', () => {
            const filter = { parentId: 'folder-1', tags: [applicationPipelineTag] };
            const gen: SagaIterator = executeFetchTemplatesAction({
                payload: filter,
                type: 'whatever',
            });
            expect(gen.next().value).toStrictEqual(call(httpGET, `/settings/features/xlrelease.DevOpsDeploymentsFeature`));
            expect(gen.next({ data: { applicationPipelineTag } }).value).toStrictEqual(call(httpPOST, `releases/templates/search`, filter));
            expect(gen.next({ data: { cis } }).value).toStrictEqual(put(setTemplates(cis)));
            expect(gen.next().done).toBeTruthy();
        });
    });

    describe('createReleaseAction', () => {
        it('should create release from a template', () => {
            const template: Template = {
                title: 'my template',
                flag: 'my flag',
                tags: ['tag1'],
                description: 'my description',
                scriptUsername: 'scriptUsername',
                scriptUserPassword: 'scriptUserPassword',
                syntheticProperties: 'scriptUserPassword',
                variableMapping: 'variableMapping',
                id: 'Applications/Folder123/Template1',
                variables: [],
            };
            const gen: SagaIterator = createReleaseAction({ payload: template, type: 'whatever' });
            expect(gen.next().value).toMatchObject(
                call(httpPOST, 'releases', {
                    abortOnFailure: false,
                    allowPasswordsInAllFields: false,
                    archiveRelease: false,
                    autoStart: true,
                    description: 'my description',
                    disableNotifications: false,
                    // TODO how to test this? dueDate: '2023-04-14T09:33:01.612Z',
                    flag: 'my flag',
                    owner: {
                        fullName: 'Release Administrator',
                        username: 'admin',
                    },
                    parentId: 'Applications/Folder123',
                    plannedDuration: null,
                    // TODO how to test this?  scheduledStartDate: '2023-04-14T09:33:01.612Z',
                    scriptUserPassword: 'scriptUserPassword',
                    scriptUsername: 'scriptUsername',
                    syntheticProperties: 'scriptUserPassword',
                    tags: ['tag1'],
                    templateId: 'Applications/Folder123/Template1',
                    title: 'my template',
                    variableMapping: 'variableMapping',
                    variables: [],
                }),
            );
            expect(gen.next({ data: { id: 'MyResponseId' } }).value).toStrictEqual(put(navigateTo({ pathSuffix: `stepper/MyResponseId` })));
            expect(gen.next().done).toBeTruthy();
        });
    });
});
