package com.xebialabs.xlrelease.applications.management.service

import com.xebialabs.deployit.checks.Checks.checkArgument
import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.xlrelease.api.internal.views.{ExternalDeploymentOrderDirection, ExternalDeploymentOrderMode}
import com.xebialabs.xlrelease.applications.management.repository.persistence.ApplicationsManagementPersistence
import com.xebialabs.xlrelease.repository.ConfigurationRepository
import com.xebialabs.xlrelease.status.service.{ConnectionServerData, EndpointExternalDeploymentService}
import com.xebialabs.xlrelease.status.webhook.configuration.StatusHttpConnection
import com.xebialabs.xlrelease.status.webhook.events.{StatusWebhookEventSource, UpdateStatusEvent}
import grizzled.slf4j.Logging
import org.springframework.stereotype.Service
import org.springframework.util.StringUtils.hasText

import java.lang
import java.util.Date

trait ApplicationsManagementService {
  def getManagedApplication(id: String): ManagedApplicationData

  def getManagedApplications(folderId: String, page: lang.Long, resultsPerPage: lang.Long, condition: String = ""): ApplicationsManagement

  def addManagedApplication(managedApplication: ManagedApplication): ManagedApplicationData

  def updateManagedApplication(managedApplication: ManagedApplication): ManagedApplicationData

  def deleteManagedApplication(managedApplicationId: String): Unit

  def createManagedApplicationsFromWebhookSource(webhookSourceId: String): Unit
}

@Service
class ApplicationManagementServiceImpl(
                                        configurationRepository: ConfigurationRepository,
                                        applicationsManagementPersistence: ApplicationsManagementPersistence,
                                        externalDeploymentService: EndpointExternalDeploymentService
                                      ) extends ApplicationsManagementService with Logging {
  override def getManagedApplications(folderId: String, page: lang.Long, resultsPerPage: lang.Long, condition: String): ApplicationsManagement = {
    val totalCount = applicationsManagementPersistence.countForFolder(folderId)
    val count = applicationsManagementPersistence.countForFolderWithCondition(folderId, condition)
    val managedApplications =
      applicationsManagementPersistence.findForFolder(folderId, resultsPerPage, resultsPerPage * page, condition).map(ma => {
        managedApplicationToManagedApplicationData(ma)
      })
    ApplicationsManagement(totalCount, count, managedApplications)
  }

  override def getManagedApplication(managedApplicationId: String): ManagedApplicationData = {
    managedApplicationToManagedApplicationData(fetchManagedApplication(managedApplicationId))
  }

  private def fetchManagedApplication(managedApplicationId: String) = {
    applicationsManagementPersistence.findById(managedApplicationId).getOrElse(
      throw new NotFoundException(s"Managed Application with id [$managedApplicationId] not found")
    )
  }

  override def addManagedApplication(managedApplication: ManagedApplication): ManagedApplicationData = {
    validateManagedApplication(managedApplication)
    val managedApplicationId = applicationsManagementPersistence.create(managedApplication)
    val created = fetchManagedApplication(managedApplicationId)
    managedApplicationToManagedApplicationData(created)
  }

  override def updateManagedApplication(managedApplication: ManagedApplication): ManagedApplicationData = {
    checkArgument(hasText(managedApplication.id), "ID is required")
    validateManagedApplication(managedApplication)
    applicationsManagementPersistence.update(managedApplication)
    managedApplicationToManagedApplicationData(managedApplication)
  }

  override def deleteManagedApplication(managedApplicationId: String): Unit = {
    checkArgument(hasText(managedApplicationId), "ID is required")
    applicationsManagementPersistence.delete(managedApplicationId)
  }

  private def validateManagedApplication(managedApplication: ManagedApplication): Unit = {
    checkArgument(hasText(managedApplication.applicationName), "Application name cannot be blank")
    checkArgument(Option(managedApplication.connectionServer).isDefined, "Connection server must be defined")
  }

  private def managedApplicationToManagedApplicationData(managedApplication: ManagedApplication): ManagedApplicationData = {
    val connection = configurationRepository.read[StatusHttpConnection](applicationsManagementPersistence.findConnectionId(managedApplication.id).get)
    ManagedApplicationData(
      managedApplication.id,
      managedApplication.applicationName,
      managedApplication.managedBy,
      managedApplication.dateCreated,
      managedApplication.environmentTag,
      managedApplication.applicationReference,
      ConnectionServerData(connection.getId, connection.getTitle, connection.getType.toString, connection.getUrl, None),
      managedApplication.workflowId,
      managedApplication.deleteWorkflowId,
      managedApplication.updateWorkflowId)
  }

  override def createManagedApplicationsFromWebhookSource(webhookSourceId: String): Unit = {
    val connection = configurationRepository.read[StatusWebhookEventSource](webhookSourceId)
    val externalDeployment = externalDeploymentService.getExternalDeployment(
      Vector(webhookSourceId),
      0,
      0,
      0,
      ExternalDeploymentOrderMode.APPLICATION,
      ExternalDeploymentOrderDirection.ASC)

    externalDeployment.states
      .map(state => ManagedApplicationFromExternalDeployment(state.event.applicationName, getApplicationReference(state.event, connection)))
      .groupBy(_.applicationReference)
      .map { case (ref, apps) => ref -> apps.head }
      .foreach { case (_, application) =>
        if (!applicationsManagementPersistence.existsByApplicationReference(application.applicationReference)) {
          addManagedApplication(
            ManagedApplication(
              null,
              application.applicationName,
              "DISCOVERED BY RELEASE",
              new Date(),
              "N/A",
              application.applicationReference,
              connection.getSourceServer().getId,
              null,
              null,
              null
            )
          )
        }
      }
  }

  private def getApplicationReference(event: UpdateStatusEvent, connection: StatusWebhookEventSource): String = {
    s"${event.applicationName}:${connection.sourceServer.getId}:[${event.destination}]"
  }
}

case class ApplicationsManagement(totalCount: Int, count: Int, managedApplications: Vector[ManagedApplicationData])

case class ManagedApplicationData(id: String,
                                  applicationName: String,
                                  managedBy: String,
                                  dateCreated: Date,
                                  environmentTag: String,
                                  applicationReference: String,
                                  connectionServer: ConnectionServerData,
                                  workflowId: String,
                                  deleteWorkflowId: String,
                                  updateWorkflowId: String,
                                 )

case class ManagedApplication(id: String,
                              applicationName: String,
                              managedBy: String,
                              dateCreated: Date,
                              environmentTag: String,
                              applicationReference: String,
                              connectionServer: String,
                              workflowId: String,
                              deleteWorkflowId: String,
                              updateWorkflowId: String,
                             )

case class ManagedApplicationFromExternalDeployment(applicationName: String, applicationReference: String)
