import { all, call, CallEffect, debounce, put, race, select, take, takeEvery, takeLatest } from 'redux-saga/effects';
import { IHttpResponse } from 'angular';
import { IStateService } from 'angular-ui-router';
import axios, { AxiosError } from 'axios';
import { PayloadAction } from '@reduxjs/toolkit';
import { CustomConfiguration, Variable } from '@xlr-ui/app/types';
import ToastrFactory from '@xlr-ui/app/js/util/toastrFactory';
import { httpDELETE, httpGET, httpPOST } from '@xlr-ui/app/features/common/services/http';
import { withFlagChangingState } from '@xlr-ui/app/react/utils/saga-utils';
import { Page } from '@xlr-ui/app/js/http/backend';
import {
    ConfigureDeploymentServer,
    DeploymentServerSearch,
    DeploymentServerState,
    folderDeploymentServers,
    getDeploymentServerState,
} from './deployment-server.reducer';
import { ConnectionServerMetadata, DeploymentServer, LiveDeploymentConfigData } from '../deployment-server.types';
import { mapConfigOrderByValue } from '../helper/utils';
import getAngularService from '@xlr-ui/app/features/common/services/angular-accessor';
import { Server, StatusWebhookEventSource } from '../../external-deployments/external-deployment.types';
import { CiTypeDescriptor } from '@xlr-ui/app/features/configuration/types';
import { STATUS_HTTP_CONNECTION, STATUS_WEBHOOK_EVENT_SOURCE_TYPE } from '../../external-deployments/constants';
import { getConnectionServerMetadata } from './deployment-server.selectors';
import { createOrUpdateVariable, getReleaseVariables, updateReleaseVariable } from '@xlr-ui/app/features/tasks/ducks/variables-service.saga';
import { RunWorkflowAction, workflow } from '@xlr-ui/app/features/workflow/ducks/workflow.reducer';
import { AUTH_ERROR_MESSAGE, CREATE_DEPLOYMENT_PROVIDER_WORKFLOWS_TITLE, DEFAULT_ADD_DEPLOYMENT_PROVIDER_SEARCH_CATEGORY } from '../constants';
import IdsFactory from '@xlr-ui/app/js/util/ids';
import {
    DEPLOYMENT_PROVIDER_APP_TAG,
    DEPLOYMENT_PROVIDER_CONFIG_TAG,
    DEPLOYMENT_PROVIDER_DELETE_TAG,
    DEPLOYMENT_PROVIDER_NEW_TAG,
    DEPLOYMENT_PROVIDER_TAG,
    INTERNAL_WORFKFLOW_TAG,
} from '@xlr-ui/app/features/workflow/constants';
import { searchWorkflowIdByTags } from '@xlr-ui/app/features/workflow/ducks/workflow.saga';

const {
    filterDeploymentServers,
    initDrawer,
    loadConnectionServerMetadata,
    loadLiveDeploymentsConfiguration,
    openDeploymentServerDrawer,
    openLiveDeploymentConfiguration,
    openLiveDeployments,
    reloadLiveDeploymentsConfiguration,
    runAddDeploymentServer,
    runDeleteDeploymentServer,
    runEditDeploymentServer,
    runUseDeploymentServer,
    setConnectionServerMetadata,
    setConfigurationCondition,
    setConfigurationPage,
    setDeploymentServers,
    setDeploymentServerSearch,
    setDialogError,
    setIsLoading,
    setLiveDeploymentConfigs,
    setLiveDeploymentConfigsCount,
    runDeleteDeploymentServerApplication,
} = folderDeploymentServers.actions;

const { createWorkflowError, createWorkflowSuccess, initCondenseViewDrawer, runWorkflow } = workflow.actions;

export const toaster = ToastrFactory();
const Ids = IdsFactory();

export type DeploymentServerMap = Record<string, StatusWebhookEventSource>;
export type ConnectionServerMap = Record<string, ConnectionServerMetadata>;

export function* initSaga(action: PayloadAction<Partial<DeploymentServerSearch> | undefined>) {
    const payloadSearch = action.payload;
    const deploymentServerSearch = {
        ...payloadSearch,
    };

    const connectionServerTypes: ConnectionServerMetadata[] = yield call(findConnectionServerMetadata);
    yield put(setConnectionServerMetadata(connectionServerTypes));
    yield call(searchDeploymentServers, deploymentServerSearch);

    const { deploymentServers } = yield select(getDeploymentServerState);
    if (deploymentServers.length > 0) {
        yield put(openDeploymentServerDrawer());
    } else {
        yield put(
            initCondenseViewDrawer({
                folderId: deploymentServerSearch.folderId as string,
                catalogTitle: CREATE_DEPLOYMENT_PROVIDER_WORKFLOWS_TITLE,
                workflowSearch: { categories: [DEFAULT_ADD_DEPLOYMENT_PROVIDER_SEARCH_CATEGORY] },
            }),
        );
    }
    yield put(setIsLoading(false));
}

export function* getFilteredDeploymentServers(sourceToServerMap: DeploymentServerMap, deploymentServerSearch?: DeploymentServerSearch) {
    const { data: servers }: IHttpResponse<Server[]> = yield call(httpPOST, `api/v1/config/byIds`, Object.keys(sourceToServerMap));
    const connectionServerTypes: ConnectionServerMetadata[] = yield select(getConnectionServerMetadata);
    const connectionServerTypeMap = connectionServerTypes.reduce((acc: ConnectionServerMap, item: ConnectionServerMetadata) => {
        acc[item.type] = item;
        return acc;
    }, {});
    const deploymentServers = servers.map((server) => {
        const source = sourceToServerMap[server.id as string];
        const metadata = connectionServerTypeMap[server.type];
        const deploymentServer: DeploymentServer = {
            connectionIconLocation: metadata.iconLocation,
            connectionId: server.id as string,
            connectionLabel: metadata.title,
            connectionTitle: server.title,
            connectionType: metadata.type,
            connectionUrl: server.url,
            eventSourceId: source.id,
            eventSourceTitle: source.title,
        };
        return deploymentServer;
    });
    return deploymentServers.filter((server) => {
        const searchInput = deploymentServerSearch?.searchInput?.toLowerCase();
        const connectionServers = deploymentServerSearch?.connectionServers;
        const connectionTitle = server.connectionTitle.toLowerCase();
        const connectionLabel = server.connectionLabel?.toLowerCase();
        const connectionUrl = server.connectionUrl.toLowerCase();
        const connectionType = server.connectionType.toLowerCase();
        const eventSourceTitle = server.eventSourceTitle.toLowerCase();
        return (
            (!searchInput ||
                connectionTitle.includes(searchInput) ||
                connectionLabel?.includes(searchInput) ||
                connectionUrl.includes(searchInput) ||
                connectionType.includes(searchInput) ||
                eventSourceTitle.includes(searchInput)) &&
            (!connectionServers || connectionServers.length === 0 || connectionServers.includes(server.connectionType))
        );
    });
}

export function* searchDeploymentServers(deploymentServerSearch?: DeploymentServerSearch) {
    const { folderId } = deploymentServerSearch || {};
    yield put(setDeploymentServerSearch(deploymentServerSearch));
    const { data: configs }: IHttpResponse<Array<StatusWebhookEventSource>> = yield call(
        httpGET,
        `api/v1/config/byTypeAndTitle?configurationType=${STATUS_WEBHOOK_EVENT_SOURCE_TYPE}&folderId=${folderId}&folderOnly=true`,
    );
    const sourceToServerMap = configs.reduce((acc: DeploymentServerMap, item: StatusWebhookEventSource) => {
        acc[item.sourceServer] = item;
        return acc;
    }, {});
    Array.from(configs.map((c) => c.sourceServer));

    const filteredDeploymentServers: DeploymentServer[] = yield call(getFilteredDeploymentServers, sourceToServerMap, deploymentServerSearch);
    yield put(setDeploymentServers(filteredDeploymentServers));
}

export function* withLoadingState<R>(effect: CallEffect) {
    try {
        yield put(setIsLoading(true));
        const result: R = yield effect;
        return result;
    } finally {
        yield put(setIsLoading(false));
    }
}

export function* executeLoadLiveDeploymentConfigurationsAction() {
    yield call(withFlagChangingState, call(executeLoadLiveDeploymentsConfiguration), setIsLoading);
}

export function* executeLoadLiveDeploymentsConfiguration() {
    const { configurationPage, configurationCondition }: DeploymentServerState = yield select(getDeploymentServerState);
    if (!configurationPage || !configurationPage.folderId) {
        return;
    }
    try {
        const { data: configs }: IHttpResponse<Page<LiveDeploymentConfigData>> = yield call(
            httpGET,
            `live-deployment/configs?folderId=${configurationPage.folderId}&page=${configurationPage.page}&resultsPerPage=${
                configurationPage.resultsPerPage
            }&orderBy=${mapConfigOrderByValue(configurationPage.orderBy)}&order=${configurationPage.order.toUpperCase()}&condition=${encodeURIComponent(
                configurationCondition,
            )}`,
            true,
        );
        yield put(setLiveDeploymentConfigs(configs.content));
        yield put(setLiveDeploymentConfigsCount(configs.totalElements));
    } catch (e: unknown) {
        if (axios.isAxiosError(e)) {
            const err = e as AxiosError<CustomConfiguration, unknown>;
            const errServerData = err.response?.data;
            const errorMessage =
                'Error fetching status data. Check connection to ' + errServerData?.title + ' HTTP Connection or check application logs for more details.';
            yield call(toaster.error, errorMessage);
        }
    }
}

export function* findConnectionServerMetadata() {
    const { data: temp }: IHttpResponse<Array<CiTypeDescriptor>> = yield call(withLoadingState, call(httpGET, `metadata/type`));

    const filteredObjects = temp.filter((object) => object.superTypes.includes(STATUS_HTTP_CONNECTION));
    return filteredObjects.map((filteredObj) => {
        const iconLocation = (filteredObj.properties.find((obj) => obj.name === 'iconLocation')?.default as string) || '';
        const subheader = (filteredObj.properties.find((obj) => obj.name === 'serverCardSubheader')?.default as string) || '';
        const title = (filteredObj.properties.find((obj) => obj.name === 'serverCardTitle')?.default as string) || '';

        return {
            subheader,
            title,
            iconLocation,
            type: filteredObj.type,
        };
    });
}

export function* deploymentServerConfigurationSaga(action: PayloadAction<ConfigureDeploymentServer>) {
    const { configurationId, connectionLabel, eventSourceId, folderId } = action.payload;
    const workflowId: string = yield call(searchWorkflowIdByTags, [
        connectionLabel.toLowerCase(),
        DEPLOYMENT_PROVIDER_TAG,
        DEPLOYMENT_PROVIDER_CONFIG_TAG,
        INTERNAL_WORFKFLOW_TAG,
    ]);
    const variables: Array<Variable> = yield call(getReleaseVariables, workflowId, true);

    const updatedWorkflowVariables: Array<Variable> = [
        yield call(createOrUpdateVariable, workflowId, variables, 'statusWebhookEventSourceId', eventSourceId, 'xlrelease.StringVariable'),
    ];

    if (configurationId) {
        const configurationVariable: Variable = yield call(
            createOrUpdateVariable,
            workflowId,
            variables,
            'liveDeploymentConfigurationId',
            configurationId,
            'xlrelease.StringVariable',
        );
        updatedWorkflowVariables.push(configurationVariable);
    }

    yield put(runWorkflow({ folderId: Ids.toDomainId(folderId), workflow: { id: workflowId } } as never as RunWorkflowAction));
    // wait until the workflow creation is finished and then clean the template variable
    yield race({
        error: take(createWorkflowError.type),
        success: take(createWorkflowSuccess.type),
    });

    yield all(updatedWorkflowVariables.map((v) => call(updateReleaseVariable, { ...v, value: '' })));
}

export function* runEditDeploymentServerSaga(action: PayloadAction<ConfigureDeploymentServer>) {
    try {
        yield call(deploymentServerConfigurationSaga, action);
    } catch (e) {
        let errorMessage = (e as Error).message;
        if (axios.isAxiosError(e) && e.response?.status === 403) {
            errorMessage = AUTH_ERROR_MESSAGE;
        }
        yield call(toaster.error, errorMessage);
    }
}

export function* runDeleteDeploymentServerSaga(action: PayloadAction<ConfigureDeploymentServer>) {
    const { configurationId, connectionLabel, eventSourceId, folderId } = action.payload;
    try {
        if (eventSourceId) {
            const workflowId: string = yield call(searchWorkflowIdByTags, [
                connectionLabel.toLowerCase(),
                DEPLOYMENT_PROVIDER_TAG,
                DEPLOYMENT_PROVIDER_DELETE_TAG,
                INTERNAL_WORFKFLOW_TAG,
            ]);
            const variables: Array<Variable> = yield call(getReleaseVariables, workflowId, true);

            const updatedWorkflowVariables: Array<Variable> = [
                yield call(createOrUpdateVariable, workflowId, variables, 'statusWebhookEventSourceId', eventSourceId, 'xlrelease.StringVariable'),
                yield call(
                    createOrUpdateVariable,
                    workflowId,
                    variables,
                    'liveDeploymentConfigurationId',
                    configurationId as string,
                    'xlrelease.StringVariable',
                ),
            ];

            yield put(runWorkflow({ folderId: Ids.toDomainId(folderId), workflow: { id: workflowId } } as never as RunWorkflowAction));
            // wait until the workflow creation is finished and then clean the template variable
            yield race({
                error: take(createWorkflowError.type),
                success: take(createWorkflowSuccess.type),
            });

            yield all(updatedWorkflowVariables.map((v) => call(updateReleaseVariable, { ...v, value: '' })));
        } else {
            yield call(httpDELETE, `api/v1/config/${configurationId}`);
            yield call(executeLoadLiveDeploymentConfigurationsAction);
        }
    } catch (e) {
        let errorMessage = (e as Error).message;
        if (axios.isAxiosError(e) && e.response?.status === 403) {
            errorMessage = AUTH_ERROR_MESSAGE;
        }
        yield call(toaster.error, errorMessage);
    }
}

export function* runUseDeploymentServerSaga(action: PayloadAction<ConfigureDeploymentServer>) {
    try {
        yield call(deploymentServerConfigurationSaga, action);
    } catch (e) {
        yield call(handleError, e);
    }
}

export function* runAddDeploymentServerSaga(action: PayloadAction<string>) {
    const folderId = action.payload;
    try {
        const workflowId: string = yield call(searchWorkflowIdByTags, [DEPLOYMENT_PROVIDER_TAG, DEPLOYMENT_PROVIDER_NEW_TAG]);
        yield put(runWorkflow({ folderId: Ids.toDomainId(folderId), workflow: { id: workflowId } } as never as RunWorkflowAction));
    } catch (e) {
        yield call(handleError, e);
    }
}

export function* runDeleteDeploymentServerApplicationSaga(action: PayloadAction<ConfigureDeploymentServer>) {
    const { folderId, configurationId: liveDeploymentId, connectionLabel } = action.payload;
    try {
        const workflowId: string = yield call(searchWorkflowIdByTags, [
            connectionLabel.toLowerCase(),
            DEPLOYMENT_PROVIDER_APP_TAG,
            DEPLOYMENT_PROVIDER_DELETE_TAG,
            INTERNAL_WORFKFLOW_TAG,
        ]);
        const variables: Array<Variable> = yield call(getReleaseVariables, workflowId, true);

        const updatedWorkflowVariables: Array<Variable> = [
            yield call(createOrUpdateVariable, workflowId, variables, 'liveDeploymentId', liveDeploymentId as string, 'xlrelease.StringVariable'),
        ];

        yield put(runWorkflow({ folderId: Ids.toDomainId(folderId), workflow: { id: workflowId } } as never as RunWorkflowAction));
        // wait until the workflow creation is finished and then clean the template variable
        yield race({
            error: take(createWorkflowError.type),
            success: take(createWorkflowSuccess.type),
        });

        yield all(updatedWorkflowVariables.map((v) => call(updateReleaseVariable, { ...v, value: '' })));
    } catch (e) {
        let errorMessage = (e as Error).message;
        if (axios.isAxiosError(e) && e.response?.status === 403) {
            errorMessage = AUTH_ERROR_MESSAGE;
        }
        yield call(toaster.error, errorMessage);
    }
}

export function* handleError(e: unknown) {
    if (axios.isAxiosError(e)) {
        const err = e as AxiosError;
        const status = err.response?.status ?? 400; // default to 400.... shouldn't happen
        yield put(setDialogError({ error: err.response?.data as string, status }));
    } else {
        const err = e as Error;
        yield put(setDialogError({ error: err.message as string, status: undefined }));
    }
}

export function* fetchConnectionServerMetadata() {
    const connectionServerMetadata: ConnectionServerMetadata[] = yield call(findConnectionServerMetadata);
    yield put(setConnectionServerMetadata(connectionServerMetadata));
}

export function* filterDeploymentServersSaga(action: PayloadAction<DeploymentServerSearch>) {
    const deploymentServerSearch = action.payload;
    yield call(searchDeploymentServers, deploymentServerSearch);
}

export function* setConfigurationConditionAction() {
    yield call(executeLoadLiveDeploymentConfigurationsAction);
}

export function* setConfigurationPageAction() {
    yield call(executeLoadLiveDeploymentConfigurationsAction);
}

export function* goToLiveDeploymentConfiguration(action: PayloadAction<boolean>) {
    const $state: IStateService = yield call(getAngularService, '$state');
    $state.go('folders.detail.live-deployment-config', { forceSetup: action.payload });
}

export function* goBackToLiveDeployments() {
    const $state: IStateService = yield call(getAngularService, '$state');
    $state.go('folders.detail.external-deployments');
}

export default function* deploymentServerRootSaga() {
    yield all([
        takeLatest(initDrawer, initSaga),
        debounce(300, filterDeploymentServers, filterDeploymentServersSaga),
        takeEvery(loadConnectionServerMetadata, fetchConnectionServerMetadata),
        takeEvery(runAddDeploymentServer, runAddDeploymentServerSaga),
        takeEvery(runUseDeploymentServer, runUseDeploymentServerSaga),
        takeEvery(runEditDeploymentServer, runEditDeploymentServerSaga),
        takeEvery(runDeleteDeploymentServer, runDeleteDeploymentServerSaga),
        takeEvery(openLiveDeploymentConfiguration, goToLiveDeploymentConfiguration),
        takeEvery(openLiveDeployments, goBackToLiveDeployments),
        takeEvery(loadLiveDeploymentsConfiguration, executeLoadLiveDeploymentConfigurationsAction),
        takeEvery(reloadLiveDeploymentsConfiguration, executeLoadLiveDeploymentConfigurationsAction),
        takeEvery(setConfigurationCondition, setConfigurationConditionAction),
        takeEvery(setConfigurationPage, setConfigurationPageAction),
        takeEvery(runDeleteDeploymentServerApplication, runDeleteDeploymentServerApplicationSaga),
    ]);
}
