import { createSlice, PayloadAction } from '@reduxjs/toolkit';
import noop from 'lodash/noop';
import { Folder } from '@xlr-ui/app/types';
import {
    Application,
    Environment,
    ExternalDeploymentEvent,
    ExternalDeploymentPage,
    LiveDeploymentData,
    LiveDeploymentsPreconditionFilter,
    PluginInfo,
    SaveWebhookSourceFiltersRequest,
    Server,
    SetupLiveUpdateRequest,
    StatusWebhookEventSource,
    WebhookEndpoint,
    WebhookSource,
    WebhookSourceAutoConfigDetails,
    WebhookSourceFilter,
} from '../external-deployment.types';

import { DEFAULT_CONDITION, DEFAULT_MAX_AGE, DEFAULT_ORDER, DEFAULT_ORDER_BY, DEFAULT_PAGE, DEFAULT_RESULTS_PER_PAGE } from '../constants';
import { ConfigurationCallbackPayload } from '@xlr-ui/app/features/configuration/types';

export interface FolderExternalDeploymentsState {
    applications: Map<string, Application>;
    condition: string;
    configDetails?: WebhookSourceAutoConfigDetails;
    connectionErrors: Array<string>;
    connectionServers: Map<string, Server>;
    count: number;
    disconnectedServers: Array<string>;
    environments: Map<string, Environment>;
    filterEventSource?: WebhookEndpoint;
    folderId?: string;
    isLoading: boolean;
    isTableView: boolean;
    liveDeployments: Array<LiveDeploymentData>;
    maxAge: number;
    page: ExternalDeploymentPage;
    preSelectedServer?: Server;
    servers: Array<Server>;
    statusWebhookEventSources: Array<StatusWebhookEventSource>;
    validServerCards: Array<PluginInfo>;
    webhookSourceFilters?: WebhookSourceFilter;
    webhookSources: Array<WebhookSource>;
}

export const initialPage: ExternalDeploymentPage = {
    folderId: undefined,
    order: DEFAULT_ORDER,
    orderBy: DEFAULT_ORDER_BY,
    page: DEFAULT_PAGE,
    resultsPerPage: DEFAULT_RESULTS_PER_PAGE,
};

export const initialState: FolderExternalDeploymentsState = {
    applications: new Map<string, Application>(),
    condition: DEFAULT_CONDITION,
    configDetails: undefined,
    connectionErrors: [],
    connectionServers: new Map<string, Server>(),
    count: 0,
    disconnectedServers: [],
    environments: new Map<string, Environment>(),
    filterEventSource: undefined,
    folderId: undefined,
    isLoading: false,
    isTableView: false,
    liveDeployments: [],
    maxAge: DEFAULT_MAX_AGE,
    page: initialPage,
    preSelectedServer: undefined,
    servers: [],
    statusWebhookEventSources: [],
    validServerCards: [],
    webhookSourceFilters: undefined,
    webhookSources: [],
};

export const folderExternalDeployments = createSlice({
    name: 'folderExternalDeployments',
    initialState,
    reducers: {
        addServerToServers: (state: FolderExternalDeploymentsState, action: PayloadAction<Server>) => {
            const serverIndex = state.servers.findIndex((s) => s.id === action.payload.id);
            if (serverIndex === -1) state.servers = [...state.servers, action.payload];
            else state.servers[serverIndex] = action.payload;
        },
        createApplicationStatus: (_state: FolderExternalDeploymentsState, _action: PayloadAction<ExternalDeploymentEvent>) => noop(),
        updateServerList: (_state: FolderExternalDeploymentsState, _action: PayloadAction<ConfigurationCallbackPayload>) => noop(),
        addValidPluginToArray: (state: FolderExternalDeploymentsState, action: PayloadAction<PluginInfo>) => {
            if (!state.validServerCards.some((plugin) => plugin.type === action.payload.type)) {
                state.validServerCards.push(action.payload);
            }
        },
        getAvailablePlugins: (_state: FolderExternalDeploymentsState, _action: PayloadAction<string>) => noop(),
        createServer: (_state: FolderExternalDeploymentsState, _action: PayloadAction<Server>) => noop(),
        deleteApplicationStatus: (_state: FolderExternalDeploymentsState, _action: PayloadAction<ExternalDeploymentEvent>) => noop(),
        deleteWebhookSource: (_state: FolderExternalDeploymentsState, _action: PayloadAction<string>) => noop(),
        getFilters: (state: FolderExternalDeploymentsState, action: PayloadAction<string>) => {
            state.page.folderId = action.payload;
        },
        loadExternalDeployments: (_state: FolderExternalDeploymentsState) => noop(),
        loadFilterEventSource: (_state: FolderExternalDeploymentsState, _action: PayloadAction<string>) => noop(),
        loadServers: (_state: FolderExternalDeploymentsState, _action: PayloadAction<string>) => noop(),
        loadWebhookSourceFilters: (_state: FolderExternalDeploymentsState, _action: PayloadAction<string>) => noop(),
        loadWebhookSources: (_state: FolderExternalDeploymentsState, _action: PayloadAction<string>) => noop(),
        patchExternalDeployments: (_state: FolderExternalDeploymentsState, _action: PayloadAction<string>) => noop(),
        refreshExternalDeployments: (_state: FolderExternalDeploymentsState) => noop(),
        removeWebhookSource: (state: FolderExternalDeploymentsState, action: PayloadAction<string>) => {
            state.webhookSources = state.webhookSources.filter((webhookSource) => webhookSource.id !== action.payload);
        },
        saveEventSource: (_state: FolderExternalDeploymentsState, _action: PayloadAction<WebhookEndpoint>) => noop(),
        setApplications: (state: FolderExternalDeploymentsState, action: PayloadAction<Map<string, Application>>) => {
            state.applications = action.payload;
        },
        setCondition: (state: FolderExternalDeploymentsState, action: PayloadAction<string>) => {
            state.condition = action.payload;
        },
        setConfigDetails: (state: FolderExternalDeploymentsState, action: PayloadAction<WebhookSourceAutoConfigDetails | undefined>) => {
            state.configDetails = action.payload;
        },
        setConnectionErrors: (state: FolderExternalDeploymentsState, action: PayloadAction<Array<string>>) => {
            state.connectionErrors = action.payload;
        },
        setConnectionServers: (state: FolderExternalDeploymentsState, action: PayloadAction<Map<string, Server>>) => {
            state.connectionServers = action.payload;
        },
        setDisconnectedServers: (state: FolderExternalDeploymentsState, action: PayloadAction<Array<string>>) => {
            state.disconnectedServers = action.payload;
        },
        setEnvironments: (state: FolderExternalDeploymentsState, action: PayloadAction<Map<string, Environment>>) => {
            state.environments = action.payload;
        },
        setFilterEventSource: (state: FolderExternalDeploymentsState, action: PayloadAction<WebhookEndpoint>) => {
            state.filterEventSource = action.payload;
        },
        setIsLoading: (state: FolderExternalDeploymentsState, action: PayloadAction<boolean>) => {
            state.isLoading = action.payload;
        },
        setIsTableView: (state: FolderExternalDeploymentsState, action: PayloadAction<boolean>) => {
            state.isTableView = action.payload;
        },
        setLiveDeployments: (state: FolderExternalDeploymentsState, action: PayloadAction<Array<LiveDeploymentData>>) => {
            state.liveDeployments = action.payload;
        },
        setLiveDeploymentsCount: (state: FolderExternalDeploymentsState, action: PayloadAction<number>) => {
            state.count = action.payload;
        },
        setLiveUpdate: (_state: FolderExternalDeploymentsState, _action: PayloadAction<SetupLiveUpdateRequest>) => noop(),
        setMaxAge: (state: FolderExternalDeploymentsState, action: PayloadAction<number>) => {
            state.maxAge = action.payload;
        },
        setPage: (state: FolderExternalDeploymentsState, action: PayloadAction<ExternalDeploymentPage>) => {
            state.page = action.payload;
        },
        setPreSelectedServer: (state: FolderExternalDeploymentsState, action: PayloadAction<Server | undefined>) => {
            state.preSelectedServer = action.payload;
        },
        setServers: (state: FolderExternalDeploymentsState, action: PayloadAction<Array<Server>>) => {
            state.servers = action.payload;
        },
        setWebhookSourceFilters: (state: FolderExternalDeploymentsState, action: PayloadAction<WebhookSourceFilter>) => {
            state.webhookSourceFilters = action.payload;
        },
        setWebhookSources: (state: FolderExternalDeploymentsState, action: PayloadAction<Array<WebhookSource>>) => {
            state.webhookSources = action.payload;
        },
        storeFilters: (_state: FolderExternalDeploymentsState) => noop(),
        subscribeToSseStream: (_state: FolderExternalDeploymentsState) => noop(),
        unsubscribeFromSseStream: (_state: FolderExternalDeploymentsState) => noop(),
        updateApplicationStatus: (_state: FolderExternalDeploymentsState, _action: PayloadAction<ExternalDeploymentEvent>) => noop(),
        updateBreadcrumbs: (_state: FolderExternalDeploymentsState, _action: PayloadAction<Folder>) => noop(),
        updateWebhookSourceFilters: (_state: FolderExternalDeploymentsState, _action: PayloadAction<SaveWebhookSourceFiltersRequest>) => noop(),
        setLiveDeploymentsPreconditionFilter: (_state: FolderExternalDeploymentsState, _action: PayloadAction<LiveDeploymentsPreconditionFilter>) => noop(),
        setStatusWebhookEventSources: (state: FolderExternalDeploymentsState, action: PayloadAction<Array<StatusWebhookEventSource>>) => {
            state.statusWebhookEventSources = action.payload;
        },
    },
});
