package com.xebialabs.xlrelease.status.webhook.events

import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.repository.ItemInUseException
import com.xebialabs.xlrelease.domain.BaseConfiguration
import com.xebialabs.xlrelease.domain.environments.LiveDeploymentConfig
import com.xebialabs.xlrelease.domain.events.ConfigurationDeletedEvent
import com.xebialabs.xlrelease.events.{AsyncSubscribe, EventListener, XLReleaseEventListener}
import com.xebialabs.xlrelease.repository.sql.persistence.configuration.ConfigurationPersistence
import com.xebialabs.xlrelease.repository.{ConfigurationRepository, PersistenceInterceptor}
import com.xebialabs.xlrelease.status.service.LiveDeploymentService
import com.xebialabs.xlrelease.status.service.script.CleanupScriptService
import com.xebialabs.xlrelease.status.webhook.configuration.StatusHttpConnection
import grizzled.slf4j.Logging
import org.springframework.stereotype.Service

import scala.jdk.CollectionConverters._

@Service
@EventListener
class StatusWebhookDeleteHandler(configurationRepository: ConfigurationRepository,
                                 configurationPersistence: ConfigurationPersistence,
                                 liveDeploymentService: LiveDeploymentService,
                                 cleanupScriptService: CleanupScriptService)
  extends XLReleaseEventListener with Logging with PersistenceInterceptor[BaseConfiguration] {

  configurationRepository.registerPersistenceInterceptor(this)

  private lazy val webhookSourceType = Type.valueOf(classOf[StatusWebhookEventSource])
  private lazy val statusHttpConnectionType = Type.valueOf(classOf[StatusHttpConnection])
  private lazy val liveDeploymentConfigType = Type.valueOf(classOf[LiveDeploymentConfig])

  @AsyncSubscribe
  def onConfigurationDelete(event: ConfigurationDeletedEvent): Unit = {
    event.conf match {
      case config: StatusWebhookEventSource =>
        statusWebhookEventSourceCleanup(config)
      case _ => // nothing
    }
  }

  override def onDelete(ciId: String): Unit = {
    configurationPersistence.getType(ciId) match {
      case Some(ciType) =>
        if (ciType.instanceOf(liveDeploymentConfigType)) checkDeploymentConfigUsedByStatusWebhookEventSource(ciId)
        if (ciType.instanceOf(statusHttpConnectionType)) checkHttpConnectionUsedByStatusWebhookEventSource(ciId)
        if (ciType.instanceOf(webhookSourceType)) startStatusWebhookEventSourceCleanupScript(ciId)
      case None =>
    }
  }

  private def statusWebhookEventSourceCleanup(source: StatusWebhookEventSource): Unit = {
    liveDeploymentService.removeStaleDeployments(source.getId)

    val referencedConfigs = configurationRepository.findAllByType[StatusWebhookEventSource](webhookSourceType).asScala
      .flatMap(_.liveDeploymentConfigs.asScala.map(_.getId))
      .toSet

    val configsToDelete = source.liveDeploymentConfigs.asScala.map(_.getId).diff(referencedConfigs)
    configsToDelete.foreach(configurationRepository.delete)
  }

  private def checkHttpConnectionUsedByStatusWebhookEventSource(ciId: String): Unit = {
    val sources = configurationRepository.findAllByType[StatusWebhookEventSource](webhookSourceType).asScala
      .filter(_.sourceServer.getId == ciId).toList

    if (sources.nonEmpty) throwItemInUseException(ciId, sources)
  }

  private def checkDeploymentConfigUsedByStatusWebhookEventSource(ciId: String): Unit = {
    val sources = configurationRepository.findAllByType[StatusWebhookEventSource](webhookSourceType).asScala
      .filter(_.liveDeploymentConfigs.asScala.map(_.getId).contains(ciId)).toList

    if (sources.nonEmpty) throwItemInUseException(ciId, sources)
  }

  private def throwItemInUseException(ciId: String, sources: List[StatusWebhookEventSource]) = {
    throw new ItemInUseException(
      s"Cannot delete configuration $ciId. It is referenced by ${sources.size} StatusWebhookEventSource(s): " +
        s"${sources.map(source => s"'${source.getId}[${source.getTitle}]'").mkString(", ")}"
    )
  }

  private def startStatusWebhookEventSourceCleanupScript(ciId: String) = {
    val source = configurationRepository.read[StatusWebhookEventSource](ciId)
    val existingConfigs = configurationRepository.findAllByType[StatusWebhookEventSource](webhookSourceType).asScala
      .filter(config => config.getId != ciId && config.sourceServer.getId == source.sourceServer.getId)
      .flatMap(_.liveDeploymentConfigs.asScala)
      .toSet
    try {
      cleanupScriptService.executeScript(source, existingConfigs)
    } catch {
      case e: Exception =>
        logger.error(s"Failed to execute cleanup script for StatusWebhookEventSource ${source.getId}", e)
        throw new ItemInUseException(
          s"Cannot delete configuration $ciId. Failed to cleanup StatusWebhookEventSource references on external system: %s", e.getMessage
        )
    }
  }
}
