package com.xebialabs.xlrelease.status.sse.service

import com.fasterxml.jackson.databind.ObjectMapper
import com.fasterxml.jackson.datatype.joda.JodaModule
import com.fasterxml.jackson.module.scala.DefaultScalaModule
import com.xebialabs.xlrelease.service.SseService
import com.xebialabs.xlrelease.status.webhook.events._
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service

trait ServerSentEventsService {
  def add(): Unit

  def send(event: LiveDeploymentEvent): Unit

  def remove(): Unit
}

@Service
class ServerSentEventsServiceImpl @Autowired()(sseService: SseService) extends ServerSentEventsService with Logging {

  private val mapper = new ObjectMapper()
  mapper.registerModule(DefaultScalaModule)
  mapper.registerModule(new JodaModule)

  private val applicationEventId = "applications-sse"

  override def add(): Unit = {
    sseService.subscribeTopicToUser(applicationEventId)
  }


  override def remove(): Unit = {
    sseService.unsubscribeTopicToUser(applicationEventId)
  }

  def send(event: LiveDeploymentEvent): Unit =
    buildSseEvent(event).foreach {
      case (eventName, content) => sseService.sendEvent(applicationEventId, eventName, content)
    }

  private def buildSseEvent(event: LiveDeploymentEvent): Option[(String, String)] =
    event match {
      case e: CreateLiveDeploymentEvent =>
        Some(("application-created", mapper.writeValueAsString(e)))
      case e: DeleteLiveDeploymentEvent =>
        Some(("application-deleted", mapper.writeValueAsString(e)))
      case e: UpdateLiveDeploymentEvent =>
        Some(("application-changed", mapper.writeValueAsString(e)))
      case _ => None
    }
}
