import React from 'react';
import moment from 'moment';
import { DotChip, DotIcon, DotTable, DotTooltip, DotTypography } from '@digital-ai/dot-components';
import { mountWithStoreAndTheme, ReactWrapper } from '../../../../../../../../../../core/xlr-ui/tests/unit/testing-utils';
import { ActiveTemplateTile, ActiveTemplateTileProps, HomeTemplate } from './active-template-tile.component';
import { primaryButton, secondaryButton } from '../../CommonCardTile/tile-buttons-row.mock';
import { CommonActiveTile } from '../../CommonCardTile/common-active-tile.component';
import { getReleaseCountText, getReleaseIdForUrl } from './helpers/helpers';
import { TemplateActionsButton } from './template-actions-button';
import * as angularAccessor from '../../../../../../../../../../core/xlr-ui/app/features/common/services/angular-accessor';
import { DATE_FORMAT_DAY_FIRST, DATE_FORMAT_MONTH_FIRST } from '../../../../../../../../../../core/xlr-ui/app/js/locale/constants';
import { calculateDotDateFormat } from '../../../../../../../../../../core/xlr-ui/app/features/tasks/components/rails/scheduling/helper';

describe('ActiveTemplateTile', () => {
    const dispatch = jest.fn();

    const title = 'Templates';
    const templates: Array<HomeTemplate> = [
        {
            releaseId: 'releaseId',
            releaseTitle: 'Test Release',
            folderName: 'test',
            folderId: 'folderId',
            folderPath: '/',
            lastEditedByUser: 1688372003150,
            releaseCount: 0,
        },
    ];

    const defaultProps: ActiveTemplateTileProps = {
        primaryButton,
        secondaryButton,
        templates,
        title,
    };

    const defaultState = {
        profile: { dateFormat: DATE_FORMAT_DAY_FIRST },
    };

    const mount = (props: ActiveTemplateTileProps = defaultProps, state = defaultState) => {
        return mountWithStoreAndTheme(<ActiveTemplateTile {...props} />, dispatch, state);
    };

    const getCommonActiveTile = (wrapper: ReactWrapper) => wrapper.find(CommonActiveTile);
    const getTable = (wrapper: ReactWrapper) => wrapper.find(DotTable);
    const getTemplateNameData = (wrapper: ReactWrapper) =>
        wrapper.findWhere((node) => node.is(DotTypography) && node.props()['data-testid'] === 'row-template-title');
    const getFolderNameData = (wrapper: ReactWrapper) =>
        wrapper.findWhere((node) => node.is(DotTypography) && node.props().className === 'template-folder-name');
    const getActionsButton = (wrapper: ReactWrapper) => wrapper.find(TemplateActionsButton);
    const getChip = (wrapper: ReactWrapper) => wrapper.find(DotChip);
    const getChipTooltip = (wrapper: ReactWrapper) =>
        wrapper.findWhere((node) => node.is(DotTooltip) && node.props()['data-testid'] === 'last-time-updated-tooltip');
    const getChipIcon = (wrapper: ReactWrapper) => wrapper.find(DotChip).find(DotIcon);

    beforeEach(() => {
        const getAngularServiceSpy = jest.spyOn(angularAccessor, 'default') as unknown as jest.SpyInstance;
        const dateFilterWrapper = (_filter: string) => (date: string, angularJsFormat: string) => moment(date).format(angularJsFormat);
        getAngularServiceSpy.mockReturnValue(dateFilterWrapper);
    });

    afterEach(() => {
        jest.resetAllMocks();
    });

    it('should render properly', () => {
        const wrapper = mount();

        const columns = [
            { id: 'templateName', label: 'Template', truncate: true },
            { id: 'lastTimeUpdated', label: 'Last updated', width: '180px' },
            { id: 'actionButton', label: '', width: '72px' },
        ];

        const commonActiveTileProps = getCommonActiveTile(wrapper).props();
        expect(commonActiveTileProps.avatarIcon).toBe('template');
        expect(commonActiveTileProps.primaryButton).toBe(primaryButton);
        expect(commonActiveTileProps.secondaryButton).toBe(secondaryButton);
        expect(commonActiveTileProps.subtitle).toBe('Last updated - Most recent 5');
        expect(commonActiveTileProps.title).toBe(title);

        const tableProps = getTable(wrapper).props();
        expect(tableProps.className).toBe('active-template-table');
        expect(tableProps.columns).toStrictEqual(columns);
        expect(tableProps.sortable).toBe(false);
    });

    it('should have correct table data', () => {
        const wrapper = mount();

        const templateName = getTemplateNameData(wrapper).at(0);
        expect(templateName.props().variant).toBe('body1');
        expect(templateName).toHaveText(templates[0].releaseTitle);

        const folderName = getFolderNameData(wrapper).at(0);
        expect(folderName.props().variant).toBe('body2');
        expect(folderName).toHaveText(`${templates[0].folderName} (${getReleaseCountText(templates[0].releaseCount)})`);

        const actionsButtonProps = getActionsButton(wrapper).at(0).props();
        expect(actionsButtonProps.selected).toBe(false);
        expect(actionsButtonProps.template).toBe(templates[0]);
        expect(actionsButtonProps.urlTemplateId).toBe(getReleaseIdForUrl(templates[0]));

        const chipProps = getChip(wrapper).props();
        expect(chipProps.size).toBe('small');
        expect(chipProps.children).toBe(moment(templates[0].lastEditedByUser).fromNow());
        expect(getChipTooltip(wrapper).props().title).toBe(
            `Last time updated ${moment(templates[0].lastEditedByUser).format(calculateDotDateFormat(DATE_FORMAT_DAY_FIRST))}`,
        );
        expect(getChipIcon(wrapper).props().iconId).toBe('calendar');
    });

    it('should display correct format based on profile settings', () => {
        const wrapper = mount(defaultProps, { ...defaultState, profile: { ...defaultState.profile, dateFormat: DATE_FORMAT_MONTH_FIRST } });
        expect(getChipTooltip(wrapper).props().title).toBe(
            `Last time updated ${moment(templates[0].lastEditedByUser).format(calculateDotDateFormat(DATE_FORMAT_MONTH_FIRST))}`,
        );
    });

    it('should handle on click for table row', () => {
        const wrapper = mount();
        const table = getTable(wrapper);
        table.invoke('onRowClick')?.({} as never, getReleaseIdForUrl(templates[0]));
        expect(window.location.href).toStrictEqual(`http://localhost/#/templates/${getReleaseIdForUrl(templates[0])}`);
    });
});
