import React from 'react';
import moment from 'moment';
import { DotTable, DotTooltip, DotTypography } from '@digital-ai/dot-components';
import { mountWithStoreAndTheme, ReactWrapper } from '../../../../../../../../../../core/xlr-ui/tests/unit/testing-utils';
import { primaryButton, secondaryButton } from '../../CommonCardTile/tile-buttons-row.mock';
import { ActiveReleaseTile, ActiveReleaseTileProps, HomeRelease } from './active-release-tile.component';
import { CommonActiveTile } from '../../CommonCardTile/common-active-tile.component';
import { getReleaseIdForUrl } from '../../TemplateTile/components/helpers/helpers';
import { DATE_FORMAT_DAY_FIRST, DATE_FORMAT_MONTH_FIRST } from '../../../../../../../../../../core/xlr-ui/app/js/locale/constants';
import * as angularAccessor from '../../../../../../../../../../core/xlr-ui/app/features/common/services/angular-accessor';
import {
    calculateDotDateFormat,
    getDaysHoursMinutesFromMS,
} from '../../../../../../../../../../core/xlr-ui/app/features/tasks/components/rails/scheduling/helper';
import { formatReleaseDuration } from './helpers/helpers';

describe('ActiveReleaseTile', () => {
    const dispatch = jest.fn();

    const title = 'Releases';
    const releases: Array<HomeRelease> = [
        {
            releaseId: 'releaseId',
            releaseTitle: 'Test Release',
            releaseStartDate: 1689771884388,
            releaseEndDate: 1690187050829,
            releaseStatus: 'in_progress',
            folderName: 'test',
            folderId: 'folderId',
            folderPath: '/',
            lastEditedByUser: 1690187050870,
        },
    ];

    const defaultProps: ActiveReleaseTileProps = {
        primaryButton,
        releases,
        secondaryButton,
        title,
    };

    const defaultState = {
        profile: { dateFormat: DATE_FORMAT_DAY_FIRST },
    };

    const mount = (props: ActiveReleaseTileProps = defaultProps, state = defaultState) => {
        return mountWithStoreAndTheme(<ActiveReleaseTile {...props} />, dispatch, state);
    };

    const getCommonActiveTile = (wrapper: ReactWrapper) => wrapper.find(CommonActiveTile);
    const getTable = (wrapper: ReactWrapper) => wrapper.find(DotTable);
    const getRowTitleData = (wrapper: ReactWrapper) =>
        wrapper.findWhere((node) => node.is(DotTypography) && node.props()['data-testid'] === 'row-release-title');
    const getRowFolderData = (wrapper: ReactWrapper) => wrapper.findWhere((node) => node.is(DotTypography) && node.props().className === 'release-folder-name');
    const getStatusTooltip = (wrapper: ReactWrapper) => wrapper.findWhere((node) => node.is(DotTooltip) && node.props()['data-testid'] === 'end-date-tooltip');
    const getRowDuration = (wrapper: ReactWrapper) =>
        wrapper.findWhere((node) => node.is(DotTypography) && node.props()['data-testid'] === 'row-release-duration');

    beforeEach(() => {
        const getAngularServiceSpy = jest.spyOn(angularAccessor, 'default') as unknown as jest.SpyInstance;
        const dateFilterWrapper = (_filter: string) => (date: string, angularJsFormat: string) => moment(date).format(angularJsFormat);
        getAngularServiceSpy.mockReturnValue(dateFilterWrapper);
    });

    afterEach(() => {
        jest.resetAllMocks();
    });

    it('should render with correct props', () => {
        const wrapper = mount();

        const commonActiveTileProps = getCommonActiveTile(wrapper).props();
        expect(commonActiveTileProps.avatarIcon).toBe('release');
        expect(commonActiveTileProps.primaryButton).toBe(primaryButton);
        expect(commonActiveTileProps.secondaryButton).toBe(secondaryButton);
        expect(commonActiveTileProps.title).toBe(title);
    });

    it('should render main content correctly', () => {
        const wrapper = mount();

        const columns = [
            { id: 'releases', label: 'Releases', truncate: true },
            { id: 'status', label: 'Status', width: '136px' },
            { id: 'endDate', label: 'End date', width: '136px' },
            { id: 'duration', label: 'Duration', width: '136px' },
        ];

        const tableProps = getTable(wrapper).props();
        expect(tableProps.className).toBe('active-release-table');
        expect(tableProps.columns).toStrictEqual(columns);
        expect(tableProps.sortable).toBe(false);
    });

    it('should have correct table data', () => {
        const wrapper = mount();

        const rowTitle = getRowTitleData(wrapper).at(0);
        expect(rowTitle.props().variant).toBe('body1');
        expect(rowTitle).toHaveText(releases[0].releaseTitle);

        const rowFolderName = getRowFolderData(wrapper).at(0);
        expect(rowFolderName.props().variant).toBe('body2');
        expect(rowFolderName).toHaveText(releases[0].folderName);

        const statusTooltip = getStatusTooltip(wrapper);
        const statusTooltipProps = statusTooltip.props();
        expect(statusTooltipProps.title).toBe(moment(releases[0].releaseEndDate).format(calculateDotDateFormat(DATE_FORMAT_DAY_FIRST)));
        expect(statusTooltipProps.placement).toBe('bottom-start');
        const endDate = statusTooltip.find(DotTypography);
        expect(endDate.props().variant).toBe('body1');
        expect(endDate).toHaveText(moment(releases[0].releaseEndDate).fromNow());

        const [durationDays, durationHours, durationMinutes] = getDaysHoursMinutesFromMS(releases[0].releaseEndDate - releases[0].releaseStartDate);

        const duration = getRowDuration(wrapper);
        expect(duration.props().variant).toBe('body1');
        expect(duration).toHaveText(formatReleaseDuration(durationDays, durationHours, durationMinutes));
    });

    it('should handle on click for table row', () => {
        const wrapper = mount();
        const table = getTable(wrapper);
        table.invoke('onRowClick')?.({} as never, getReleaseIdForUrl(releases[0]));
        expect(window.location.href).toStrictEqual(`http://localhost/#/releases/${getReleaseIdForUrl(releases[0])}`);
    });

    it('should display correct date format based on profile settings', () => {
        const wrapper = mount(defaultProps, { ...defaultState, profile: { ...defaultState.profile, dateFormat: DATE_FORMAT_MONTH_FIRST } });
        expect(getStatusTooltip(wrapper).props().title).toBe(moment(releases[0].releaseEndDate).format(calculateDotDateFormat(DATE_FORMAT_MONTH_FIRST)));
    });
});
