package com.xebialabs.xlrelease.plugins.dashboard.cache

import com.xebialabs.deployit.plugin.api.reflect.{PropertyDescriptor, PropertyKind}
import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem
import com.xebialabs.xlrelease.plugins.dashboard.domain.Tile
import com.xebialabs.xlrelease.user.User
import grizzled.slf4j.Logging
import org.springframework.cache.interceptor.{KeyGenerator, SimpleKey}

import java.lang.reflect.Method
import java.util
import scala.collection.immutable.ListMap
import scala.jdk.CollectionConverters._

class TileCacheKeyGenerator extends KeyGenerator with Logging {

  val separator = "|"
  val userSpecificCache = "userSpecificCache"
  val id = "id"
  val cacheKey = "cacheKeys"

  override def generate(target: Any, method: Method, params: AnyRef*): AnyRef = {
    val valuesForKey: Seq[String] = params.map {
      case map: Map[_, _] => map.mkString(separator)
      case tile: Tile =>
        val configurableTileProperties = tile.getProperties.asScala
        val configurationProps = BooleanFold(tile.hasProperty(cacheKey))
          .fold(configurableTileProperties) { descriptor =>
            val propertiesNames = tile.getProperty[util.Set[String]](cacheKey)
            configurableTileProperties.filter(d => propertiesNames.contains(d.getName))
          }
          .map(descriptor => {
            val propertyValue: Any = tile.getProperty(descriptor.getName)
            val token = fetchToken(descriptor, propertyValue)
            descriptor.getName -> (propertyValue + token)
          })
          .toMap
        val props: Map[String, Any] = if (tile.isUserSpecificCache) {
          Map(id -> tile.getId, userSpecificCache -> User.AUTHENTICATED_USER.getName) ++ configurationProps
        } else {
          Map(id -> tile.getId) ++ configurationProps
        }
        ListMap(props.toSeq.sortBy(_._1): _*).mkString(separator)
      case _ => ""
    }

    val key = new SimpleKey(valuesForKey.filterNot(_.isEmpty).mkString(separator))
    logger.debug(s"Generated key with hashcode: ${key.hashCode}, key: $key")
    key
  }

  private def fetchToken(descriptor: PropertyDescriptor, propertyValue: Any): String = {
    if (null != propertyValue && descriptor.getKind == PropertyKind.CI) {
      Option(propertyValue.asInstanceOf[BaseConfigurationItem].get$token()).getOrElse("")
    } else {
      ""
    }
  }

  implicit class BooleanFold(boolean: Boolean) {
    def fold[B](l: B)(r: B => B): B = if (boolean) r(l) else l
  }
}
