import React from 'react';
import { mountWithTheme, ReactWrapper } from '../../../../../../../../../../../core/xlr-ui/tests/unit/testing-utils';
import { DashboardActionContextMenu, DashboardActionContextMenuProps } from './dashboard-action-context-menu.component';
import { dashboardsMock } from '../../mocks';
import { DotMenu } from '@digital-ai/dot-components';
import { DashboardDialog } from './list/dashboard-dialog.component';

describe('DashboardActionContextMenu', () => {
    let wrapper: ReactWrapper;
    const element = document.createElement('div');
    const dashboard = dashboardsMock[0];
    const onDelete = jest.fn();
    const onDuplicate = jest.fn();
    const onLeave = jest.fn();

    const defaultProps: DashboardActionContextMenuProps = {
        anchorEl: element,
        dashboard,
        hasCreateDashboardPermission: true,
        hasDeleteDashboardPermission: true,
        onDelete,
        onDuplicate,
        onLeave,
    };

    const getDotMenu = () => wrapper.find(DotMenu);
    const getDashboardDialog = () => wrapper.find(DashboardDialog);

    const mount = (props = defaultProps) => {
        wrapper = mountWithTheme(<DashboardActionContextMenu {...props} />);
    };

    afterEach(() => jest.resetAllMocks());

    it('should render correct elements with the default props', () => {
        mount();
        const dotMenu = getDotMenu();
        expect(dotMenu).toExist();
        const dotMenuProps = dotMenu.props();
        expect(dotMenuProps.anchorEl).toStrictEqual(element);
        expect(dotMenuProps.dense).toBe(true);
        expect(dotMenuProps.disablePortal).toBe(true);
        expect(dotMenuProps.id).toBe('dashboard-context-menu');
        expect(dotMenuProps.menuItems).toHaveLength(2);
        expect(dotMenuProps.menuPlacement).toBe('bottom-end');
        expect(dotMenuProps.open).toBe(true);

        const dashboardDialog = getDashboardDialog();
        expect(dashboardDialog).not.toExist();
    });

    it('should NOT render context menu when dashboard is not defined', () => {
        mount({ ...defaultProps, dashboard: null });
        expect(getDotMenu()).not.toExist();
    });

    it('should execute correct handler when DELETE option is selected', () => {
        mount();
        const dashboardActionType = 'DELETE';
        const dotMenu = getDotMenu();
        dotMenu.invoke('onSelect')?.({} as never, {} as never, dashboardActionType);

        const dashboardDialog = getDashboardDialog();
        expect(dashboardDialog).toExist();
        const dashboardDialogProps = dashboardDialog.props();
        expect(dashboardDialogProps.dashboard).toStrictEqual(dashboard);
        expect(dashboardDialogProps.dashboardActionType).toBe(dashboardActionType);
        expect(dashboardDialogProps.dialogTitle).toBe('Delete dashboard');
    });

    it('should execute correct handler when DUPLICATE option is selected', () => {
        mount();
        const dashboardActionType = 'DUPLICATE';
        const dotMenu = getDotMenu();
        dotMenu.invoke('onSelect')?.({} as never, {} as never, 'DUPLICATE');

        const dashboardDialog = getDashboardDialog();
        expect(dashboardDialog).toExist();
        const dashboardDialogProps = dashboardDialog.props();
        expect(dashboardDialogProps.dashboard).toStrictEqual(dashboard);
        expect(dashboardDialogProps.dashboardActionType).toBe(dashboardActionType);
        expect(dashboardDialogProps.dialogTitle).toBe('Duplicate dashboard');
    });

    it('should execute correct handler when onDialogSubmit function has been triggered on the DashboardDialog with DELETE action', () => {
        mount();
        const dotMenu = getDotMenu();
        dotMenu.invoke('onSelect')?.({} as never, {} as never, 'DELETE');
        const dashboardDialog = getDashboardDialog();
        dashboardDialog.invoke('onDialogSubmit')?.({} as never);
        expect(onDelete).toHaveBeenCalledWith(dashboard.id);
    });

    it('should execute correct handler when onDialogSubmit function has been triggered on the DashboardDialog with DUPLICATE action', () => {
        mount();
        const duplicateTitle = `${dashboard.title} (copy)`;
        const dotMenu = getDotMenu();
        dotMenu.invoke('onSelect')?.({} as never, {} as never, 'DUPLICATE');
        const dashboardDialog = getDashboardDialog();
        dashboardDialog.invoke('onDialogSubmit')(duplicateTitle);
        expect(onDuplicate).toHaveBeenCalledWith({ ...dashboard, metadata: undefined, id: null, title: duplicateTitle });
    });

    it('should close confirmation modal when onCancel prop has been invoked on DotConfirmationDialog component', () => {
        mount();
        getDotMenu().invoke('onSelect')?.({} as never, {} as never, 'DUPLICATE');
        getDashboardDialog().invoke('onDialogClose')();
        expect(getDashboardDialog()).not.toExist();
    });
});
