package com.xebialabs.xlrelease.environments.repository.sql

import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.xlrelease.api.v1.filter.EnvironmentFilters
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.transaction.{IsReadOnly, IsTransactional}
import com.xebialabs.xlrelease.domain.environments.Environment
import com.xebialabs.xlrelease.environments.repository.EnvironmentRepository
import com.xebialabs.xlrelease.environments.repository.sql.persistence.builder.{ColumnAliases, EnvironmentSqlBuilder}
import com.xebialabs.xlrelease.environments.repository.sql.persistence.{EnvironmentPersistence, EnvironmentStagePersistence}
import com.xebialabs.xlrelease.repository.Page
import com.xebialabs.xlrelease.service.CiIdService
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.stereotype.Repository

import scala.jdk.CollectionConverters._

@IsTransactional
@Repository
class SqlEnvironmentRepository @Autowired()(@Qualifier("xlrRepositorySqlDialect") implicit val dialect: Dialect,
                                            implicit val ciIdService: CiIdService,
                                            environmentPersistence: EnvironmentPersistence,
                                            environmentStagePersistence: EnvironmentStagePersistence)
  extends EnvironmentRepository with Logging {

  @IsReadOnly
  override def searchEnvironments(environmentFilters: EnvironmentFilters, page: Page): Seq[Environment] = {
    val query = EnvironmentSqlBuilder()
      .select()
      .orderBy(ColumnAliases.Environments.CI_UID)
      .limitAndOffset(page.resultsPerPage, page.offset)

    Option(environmentFilters).foreach { f =>
      Option(f.getTitle).foreach(query.withTitleLike)
      Option(f.getStage).foreach(query.withStageTitle)
      Option(f.getLabels).foreach(label => query.withLabelTitles(label.asScala))
    }

    environmentPersistence.search(query.build())
  }

  @IsReadOnly
  override def findEnvironmentById(environmentId: String): Environment = {
    environmentPersistence.findById(environmentId)
      .getOrElse(throw new NotFoundException(s"Environment [$environmentId] not found"))
  }

  @IsReadOnly
  override def findEnvironmentByTitle(environmentTitle: String): Environment = {
    environmentPersistence.findByTitle(environmentTitle)
      .getOrElse(throw new NotFoundException(s"Environment [$environmentTitle] not found"))
  }

  override def createEnvironment(environment: Environment): Environment = {
    val envId = environmentPersistence.insert(environment)
    findEnvironmentById(envId)
  }

  override def updateEnvironment(environment: Environment): Environment = {
    environmentPersistence.update(environment)
    findEnvironmentById(environment.getId)
  }

  override def deleteEnvironment(environmentId: String): Unit = {
    environmentPersistence.delete(environmentId)
  }
}
