package com.xebialabs.xlrelease.api.v1.impl;

import java.util.List;
import java.util.stream.Collectors;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import com.codahale.metrics.annotation.Timed;

import com.xebialabs.xlrelease.api.v1.ApplicationApi;
import com.xebialabs.xlrelease.api.v1.filter.ApplicationFilters;
import com.xebialabs.xlrelease.api.v1.form.ApplicationForm;
import com.xebialabs.xlrelease.api.v1.views.ApplicationView;
import com.xebialabs.xlrelease.domain.environments.Application;
import com.xebialabs.xlrelease.environments.service.ApplicationSecurity;
import com.xebialabs.xlrelease.environments.service.ApplicationService;

@Controller
public class ApplicationApiImpl implements ApplicationApi {

    private final ApplicationService applicationService;
    private final ApplicationSecurity applicationSecurity;

    @Autowired
    public ApplicationApiImpl(ApplicationService applicationService, ApplicationSecurity applicationSecurity) {
        this.applicationService = applicationService;
        this.applicationSecurity = applicationSecurity;
    }

    @Timed
    @Override
    public ApplicationView createApplication(ApplicationForm applicationForm) {
        applicationSecurity.checkCreate();
        return ApplicationView.fromApplication(applicationService.createApplication(applicationForm.toApplication()));
    }

    @Timed
    @Override
    public ApplicationView getApplication(String applicationId) {
        applicationSecurity.checkView();
        return ApplicationView.fromApplication(applicationService.findApplicationById(applicationId));
    }

    @Timed
    @Override
    public ApplicationView updateApplication(String applicationId, ApplicationForm applicationForm) {
        applicationSecurity.checkEdit();
        Application application = applicationForm.toApplication();
        application.setId(applicationId);
        return ApplicationView.fromApplication(applicationService.updateApplication(application));
    }

    @Timed
    @Override
    public List<ApplicationView> searchApplications(ApplicationFilters applicationFilters) {
        applicationSecurity.checkView();
        return applicationService.searchApplications(applicationFilters)
                .stream().map(ApplicationView::fromApplication).collect(Collectors.toList());
    }

    @Timed
    @Override
    public void deleteApplication(String applicationId) {
        applicationSecurity.checkEdit();
        applicationService.deleteApplication(applicationId);
    }

    @Timed
    @Override
    public Application create(Application application) {
        applicationSecurity.checkCreate();
        return applicationService.createApplication(application);
    }

    @Timed
    @Override
    public Application getById(String applicationId) {
        applicationSecurity.checkView();
        return applicationService.findApplicationById(applicationId);
    }

    @Timed
    @Override
    public Application update(Application application) {
        applicationSecurity.checkEdit();
        return applicationService.updateApplication(application);
    }

    @Timed
    @Override
    public List<Application> search(ApplicationFilters filters) {
        applicationSecurity.checkView();
        return applicationService.searchApplications(filters);
    }
}
