package com.xebialabs.xlrelease.environments.service

import com.codahale.metrics.annotation.Timed
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlrelease.actors.ReleaseActorService
import com.xebialabs.xlrelease.domain.Task
import com.xebialabs.xlrelease.domain.events.{FacetDeletedEvent, FacetUpdatedEvent}
import com.xebialabs.xlrelease.domain.status.TaskStatus
import com.xebialabs.xlrelease.environments.service.EnvironmentReservationEventHandler.StartTaskNowIfNecessary
import com.xebialabs.xlrelease.events.{EventBus, Subscribe}
import com.xebialabs.xlrelease.repository.{Ids, TaskRepository}
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service

import jakarta.annotation.{PostConstruct, PreDestroy}
import scala.jdk.CollectionConverters._

@Service
class FacetEventHandler @Autowired()(val eventBus: EventBus,
                                     val taskRepository: TaskRepository,
                                     val releaseActorService: ReleaseActorService) {


  private lazy val DeploymentFacetType = Type.valueOf("udm.DeploymentTaskFacet")

  @PostConstruct
  def register(): Unit = {
    eventBus.register(this)
  }

  @PreDestroy
  def unregister(): Unit = {
    eventBus.deregister(this)
  }

  @Subscribe
  @Timed
  def onFacetDeleted(facetDeletedEvent: FacetDeletedEvent): Unit = {
    if (facetDeletedEvent.facet.getType == DeploymentFacetType) {
      uncheckTaskEnvironmentAndRestartTaskIfNecessary(facetDeletedEvent.facet.getTargetId)
    }
  }

  @Subscribe
  @Timed
  def onFacetUpdated(facetUpdatedEvent: FacetUpdatedEvent): Unit = {
    if (facetUpdatedEvent.facet.getType == DeploymentFacetType) {
      uncheckTaskEnvironmentAndRestartTaskIfNecessary(facetUpdatedEvent.facet.getTargetId)
    }
  }

  private def uncheckTaskEnvironmentAndRestartTaskIfNecessary(ciId: String): Unit = {
    if (Ids.isTaskId(ciId)) {
      val task = taskRepository.findById[Task](ciId)
      if (task.isCheckAttributes &&
        (task.getFacets.isEmpty || !task.getFacets.asScala.exists(_.getType == DeploymentFacetType))) {
        task.setCheckAttributes(false)
        releaseActorService.updateTask(ciId, task)
      }
      if (task.isPostponedUntilEnvironmentsAreReserved && task.getStatus == TaskStatus.PENDING) {
        val taskId = Ids.getFolderlessId(ciId)
        releaseActorService.executeCommandAsync(Ids.releaseIdFrom(taskId), StartTaskNowIfNecessary(taskId))
      }
    }
  }

}
