package com.xebialabs.xlrelease.environments.service
import com.codahale.metrics.annotation.Timed
import com.xebialabs.deployit.checks.Checks.checkArgument
import com.xebialabs.xlrelease.api.v1.filter.EnvironmentFilters
import com.xebialabs.xlrelease.domain.environments.Environment
import com.xebialabs.xlrelease.environments.events.{EnvironmentCreatedEvent, EnvironmentDeletedEvent, EnvironmentUpdatedEvent}
import com.xebialabs.xlrelease.environments.repository.EnvironmentRepository
import com.xebialabs.xlrelease.events.XLReleaseEventBus
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service
import org.springframework.util.StringUtils.hasText

import java.util.{List => JList}
import scala.jdk.CollectionConverters._

@Service
class EnvironmentService @Autowired()(environmentRepository: EnvironmentRepository,
                                      val eventBus: XLReleaseEventBus) extends Logging {

  @Timed
  def searchEnvironments(environmentFilters: EnvironmentFilters): JList[Environment] = {
    environmentRepository.searchEnvironments(environmentFilters, defaultPage).asJava
  }

  @Timed
  def findEnvironmentById(environmentId: String): Environment = {
    environmentRepository.findEnvironmentById(environmentId)
  }

  @Timed
  def findEnvironmentByTitle(environmentId: String): Environment = {
    environmentRepository.findEnvironmentByTitle(environmentId)
  }

  @Timed
  def createEnvironment(environment: Environment): Environment = {
    logger.debug("Creating a new environment")
    validate(environment)
    val created = environmentRepository.createEnvironment(environment)
    eventBus.publish(EnvironmentCreatedEvent(created))
    created
  }

  @Timed
  def updateEnvironment(environment: Environment): Environment = {
    logger.debug(s"Updating environment [$environment]")
    checkArgument(hasText(environment.getId), "ID is required")
    validate(environment)
    val updated = environmentRepository.updateEnvironment(environment)
    eventBus.publish(EnvironmentUpdatedEvent(environment))
    updated
  }

  @Timed
  def deleteEnvironment(environmentId: String): Unit = {
    logger.debug(s"Deleting environment [$environmentId]")
    val environment = environmentRepository.findEnvironmentById(environmentId)
    environmentRepository.deleteEnvironment(environmentId)
    eventBus.publish(EnvironmentDeletedEvent(environment))
  }

  private def validate(environment: Environment): Unit = {
    validateTitle(environment)
    validateDescription(environment)
    checkArgument(Option(environment.getStage).exists(stage => hasText(stage.getId)), "Environment stage ID is required")
  }

  private def validateTitle(environment: Environment): Unit = {
    checkArgument(hasText(environment.getTitle), "Title cannot be blank")
  }

  private def validateDescription(environment: Environment): Unit = {
    if (environment.getDescription != null) {
      checkArgument(hasText(environment.getTitle), "Description cannot be blank")
    }
  }
}
