package com.xebialabs.xlrelease.environments.repository.sql.persistence

import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.SqlWithParameters
import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional
import com.xebialabs.xlrelease.domain.environments.Application
import com.xebialabs.xlrelease.environments.repository.sql.persistence.builder.ApplicationSqlBuilder
import com.xebialabs.xlrelease.environments.repository.sql.persistence.schema.ApplicationSchema
import com.xebialabs.xlrelease.environments.repository.sql.persistence.schema.ApplicationSchema.{APPLICATIONS, APPLICATION_TO_ENVIRONMENTS}
import com.xebialabs.xlrelease.environments.repository.sql.persistence.schema.EnvironmentSchema.ENVIRONMENTS
import com.xebialabs.xlrelease.repository.sql.persistence.CiId.CiId
import com.xebialabs.xlrelease.repository.sql.persistence.Utils._
import com.xebialabs.xlrelease.repository.sql.persistence.{CiUid, PersistenceSupport, SecurablePersistence}
import com.xebialabs.xlrelease.service.CiIdService
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.dao.DuplicateKeyException
import org.springframework.jdbc.core.JdbcTemplate
import org.springframework.stereotype.Repository
import org.springframework.util.CollectionUtils

import scala.jdk.CollectionConverters._
import scala.util.Try


@Repository
@IsTransactional
class ApplicationPersistence @Autowired()(@Qualifier("xlrRepositoryJdbcTemplate") implicit val jdbcTemplate: JdbcTemplate,
                                          @Qualifier("xlrRepositorySqlDialect") implicit val dialect: Dialect,
                                          implicit val ciIdService: CiIdService,
                                          val environmentPersistence: EnvironmentPersistence,
                                          val securablePersistence: SecurablePersistence)
  extends PersistenceSupport {

  private val STMT_INSERT_APPLICATIONS =
    s"""|INSERT INTO ${APPLICATIONS.TABLE}
        |   ( ${APPLICATIONS.CI_UID}
        |   , ${APPLICATIONS.ID}
        |   , ${APPLICATIONS.TITLE}
        |   )
        | VALUES
        |   ( :${APPLICATIONS.CI_UID}
        |   , :${APPLICATIONS.ID}
        |   , :${APPLICATIONS.TITLE}
        |   )
        """.stripMargin

  private val STMT_EXISTS_BY_TITLE_IGNORECASE = s"SELECT COUNT(*) FROM ${APPLICATIONS.TABLE} WHERE LOWER(${APPLICATIONS.TITLE}) = LOWER(:${APPLICATIONS.TITLE})"

  def insert(application: Application): CiId = {
    sanitizeApplicationInput(application)

    val exists = sqlQuery(STMT_EXISTS_BY_TITLE_IGNORECASE, params(ENVIRONMENTS.TITLE -> application.getTitle), _.getInt(1) > 0).head
    if (exists) {
      throw new IllegalArgumentException(s"Application with title '${application.getTitle}' already exists")
    }

    val environmentUids = getEnvironmentUids(application)

    val ciUid = securablePersistence.insert()
    val appId = createPersistedId[Application]
    val displayedAppId = toDisplayId(appId)
    try {
      sqlExec(STMT_INSERT_APPLICATIONS, params(
        APPLICATIONS.CI_UID -> ciUid,
        APPLICATIONS.ID -> appId,
        APPLICATIONS.TITLE -> application.getTitle
      ), _.execute())
    } catch {
      case ex: DuplicateKeyException => throw new IllegalArgumentException(s"Application with ID '$displayedAppId' already exists", ex)
    }

    insertAppToEnvReferences(ciUid, environmentUids.valuesIterator.toSet)

    displayedAppId
  }

  private val STMT_UPDATE_APPLICATIONS =
    s"""|UPDATE ${APPLICATIONS.TABLE}
        | SET
        |  ${APPLICATIONS.TITLE} = :${APPLICATIONS.TITLE}
        | WHERE
        |  ${APPLICATIONS.CI_UID} = :${APPLICATIONS.CI_UID}
       """.stripMargin

  private val STMT_EXISTS_ANOTHER_APP_WITH_TITLE =
    s"""|SELECT COUNT(*)
        |FROM ${APPLICATIONS.TABLE}
        |WHERE LOWER(${APPLICATIONS.TITLE}) = LOWER(:${APPLICATIONS.TITLE})
        |AND ${APPLICATIONS.CI_UID} <> :${APPLICATIONS.CI_UID}
       """.stripMargin

  def update(application: Application): Boolean = {
    val applicationUid = findUidById(application.getId)
      .getOrElse(throw new NotFoundException(s"Application [${application.getId}] not found"))

    sanitizeApplicationInput(application)

    val existsAnotherWithSameTitle = sqlQuery(STMT_EXISTS_ANOTHER_APP_WITH_TITLE,
      params(APPLICATIONS.TITLE -> application.getTitle, APPLICATIONS.CI_UID -> applicationUid),
      _.getInt(1) > 0).head
    if (existsAnotherWithSameTitle) {
      throw new IllegalArgumentException(s"Application with title '${application.getTitle}' already exists")
    }

    updateAppToEnvReferences(application, applicationUid)

    sqlUpdate(
      STMT_UPDATE_APPLICATIONS,
      params(
        APPLICATIONS.CI_UID -> applicationUid,
        APPLICATIONS.TITLE -> application.getTitle
      ),
      _ == 1
    )
  }

  private val STMT_DELETE_APPLICATIONS_BY_ID =
    s"""|DELETE FROM ${APPLICATIONS.TABLE}
        | WHERE ${APPLICATIONS.ID} = :${APPLICATIONS.ID}
       """.stripMargin

  def delete(applicationId: CiId): Try[Boolean] =
    sqlExec(STMT_DELETE_APPLICATIONS_BY_ID, params(APPLICATIONS.ID -> toPersistedId(applicationId)), ps => Try(ps.execute()))

  def findById(applicationId: CiId): Option[Application] = {
    val (sql, params) = ApplicationSqlBuilder().select().withApplicationId(toPersistedId(applicationId)).build()
    jdbcTemplate.query(sql, params.toArray, Mappers.applicationResultSetExtractor())
  }

  def findByTitle(title: String): Option[Application] = {
    val (sql, params) = ApplicationSqlBuilder().select().withTitle(title).build()
    jdbcTemplate.query(sql, params.toArray, Mappers.applicationResultSetExtractor())
  }

  def findUidById(applicationId: CiId): Option[CiUid] = {
    val stmt =
      s"""|SELECT ${APPLICATIONS.CI_UID}
          |FROM ${APPLICATIONS.TABLE}
          |WHERE ${APPLICATIONS.ID} = :${APPLICATIONS.ID}""".stripMargin
    sqlQuery(stmt, params(APPLICATIONS.ID -> toPersistedId(applicationId)), rs => CiUid(rs.getInt(APPLICATIONS.CI_UID))).headOption
  }

  def getUidsByIds(applicationIds: Iterable[CiId]): Map[CiId, CiUid] = {
    val stmt =
      s"""|SELECT ${APPLICATIONS.CI_UID}, ${APPLICATIONS.ID}
          |FROM ${APPLICATIONS.TABLE}
          |WHERE ${APPLICATIONS.ID} IN (:applicationIds)
       """.stripMargin
    sqlQuery(stmt,
      params("applicationIds" -> applicationIds.map(toPersistedId).asJava),
      rs => rs.getCiId(APPLICATIONS.ID) -> CiUid(rs.getInt(APPLICATIONS.CI_UID))
    ).toMap
  }

  def search(sqlWithParameters: SqlWithParameters): Seq[Application] = {
    val (sql, params) = sqlWithParameters
    jdbcTemplate.query(sql, params.toArray, Mappers.applicationSearchResultSetExtractor)
  }

  private def getEnvironmentUids(application: Application): Map[CiId, CiUid] = {
    if (CollectionUtils.isEmpty(application.getEnvironments)) {
      Map.empty[CiId, CiUid]
    } else {
      val environmentIds = application.getEnvironments.asScala.map(_.getId).toSet
      val environmentIdsToUids = environmentPersistence.getUidsByIds(environmentIds)
      val environmentDiff = environmentIds.diff(environmentIdsToUids.keySet)
      if (environmentDiff.nonEmpty) {
        throw new NotFoundException(s"Environments [${environmentDiff.mkString(", ")}] not found")
      }
      environmentIdsToUids
    }
  }

  private def insertAppToEnvReferences(applicationUid: CiUid, envUids: Set[CiUid]): Unit = {
    val insertAppTpEnvStmt =
      s"""INSERT INTO ${APPLICATION_TO_ENVIRONMENTS.TABLE} (
         |${APPLICATION_TO_ENVIRONMENTS.APPLICATION_UID},
         |${APPLICATION_TO_ENVIRONMENTS.ENVIRONMENT_UID}
         |)
         |VALUES (
         |:${APPLICATION_TO_ENVIRONMENTS.APPLICATION_UID},
         |:${APPLICATION_TO_ENVIRONMENTS.ENVIRONMENT_UID}
         |)
       """.stripMargin

    sqlBatch(insertAppTpEnvStmt,
      envUids.map { envUid =>
        params(
          APPLICATION_TO_ENVIRONMENTS.APPLICATION_UID -> applicationUid,
          APPLICATION_TO_ENVIRONMENTS.ENVIRONMENT_UID -> envUid
        )
      }
    )
  }

  private def updateAppToEnvReferences(application: Application, applicationUid: CiUid): Unit = {
    val envIdsToUids = getEnvironmentUids(application)

    val deleteAppToEnvRefsStmt =
      s"""
         |DELETE FROM ${APPLICATION_TO_ENVIRONMENTS.TABLE}
         | WHERE
         | ${APPLICATION_TO_ENVIRONMENTS.APPLICATION_UID} = :${APPLICATION_TO_ENVIRONMENTS.APPLICATION_UID}
       """.stripMargin
    sqlExec(deleteAppToEnvRefsStmt, params(APPLICATION_TO_ENVIRONMENTS.APPLICATION_UID -> applicationUid), _.execute())
    insertAppToEnvReferences(applicationUid, envIdsToUids.valuesIterator.toSet)
  }

  private def sanitizeApplicationInput(application: Application): Unit = {
    application.setTitle(application.getTitle.trimAndTruncate(ApplicationSchema.TITLE_LENGTH))
  }
}
