package com.xebialabs.xlrelease.environments.repository.sql.persistence

import com.xebialabs.xlrelease.domain.environments._
import com.xebialabs.xlrelease.environments.repository.sql.persistence.builder.ColumnAliases
import org.springframework.jdbc.core.{ResultSetExtractor, RowMapper}

import java.util.Date
import scala.collection.mutable
import scala.jdk.CollectionConverters._

object Mappers {
  def applicationResultSetExtractor(): ResultSetExtractor[Option[Application]] = rs => {
    // No stages are fetched in this search for now
    val noopStageMapper: RowMapper[EnvironmentStage] = (_, _) => {
      null
    }
    val envMapper = environmentMapper(noopStageMapper)

    if (rs.next()) {
      var rowNum = 0
      val application = applicationMapper.mapRow(rs, rowNum)
      if (rs.getCiId(ColumnAliases.Environments.ID) != null) {
        application.getEnvironments.add(envMapper.mapRow(rs, rowNum))
        rowNum += 1
        while (rs.next()) {
          application.getEnvironments.add(envMapper.mapRow(rs, rowNum))
          rowNum += 1
        }
      }

      Some(application)
    } else {
      None
    }
  }

  def environmentResultSetExtractor(stageMapper: RowMapper[EnvironmentStage] = environmentStageMapper,
                                    labelMapper: RowMapper[EnvironmentLabel] = environmentLabelMapper
                                   ): ResultSetExtractor[Option[Environment]] = rs => {
    if (rs.next()) {
      var rowNum = 0
      val environment = environmentMapper(stageMapper).mapRow(rs, rowNum)
      if (rs.getCiId(ColumnAliases.EnvLabels.ID) != null) {
        environment.getLabels.add(labelMapper.mapRow(rs, rowNum))
        rowNum += 1
        while (rs.next()) {
          environment.getLabels.add(labelMapper.mapRow(rs, rowNum))
          rowNum += 1
        }
      }

      Some(environment)
    } else {
      None
    }
  }

  def applicationSearchResultSetExtractor: ResultSetExtractor[Seq[Application]] = rs => {
    val envMap = mutable.Map.empty[String, Environment]
    val appMap = mutable.Map.empty[String, Application]
    val appToEnvsMap = mutable.Map.empty[String, mutable.Set[Environment]]
    // No stages are fetched in this search for now
    val noopStageMapper: RowMapper[EnvironmentStage] = (_, _) => {
      null
    }

    val cachedEnvMapper: RowMapper[Environment] = (rs, rowNum) => {
      val envId = rs.getCiId(ColumnAliases.Environments.ID)
      envMap.getOrElseUpdate(envId, environmentMapper(noopStageMapper).mapRow(rs, rowNum))
    }
    val cachedAppMapper: RowMapper[Application] = (rs, rowNum) => {
      val app = cachedApplicationMapper(appMap).mapRow(rs, rowNum)

      if (rs.getCiId(ColumnAliases.Environments.ID) != null) {
        appToEnvsMap.getOrElseUpdate(app.getId, mutable.Set.empty) += cachedEnvMapper.mapRow(rs, rowNum)
      }

      app
    }

    var rowNum = 0
    while (rs.next()) {
      val appId = rs.getCiId(ColumnAliases.Applications.ID)
      if (appId != null) {
        cachedAppMapper.mapRow(rs, rowNum)
      }
      rowNum += 1
    }

    appMap.view.map { case (appId, app) =>
      app.setEnvironments(appToEnvsMap.getOrElse(appId, Set.empty[Environment]).view.toList.asJava)
      app
    }.toSeq
  }

  def environmentSearchResultSetExtractor: ResultSetExtractor[Seq[Environment]] = rs => {
    val envMap = mutable.Map.empty[String, Environment]
    val stageMap = mutable.Map.empty[String, EnvironmentStage]
    val labelMap = mutable.Map.empty[String, EnvironmentLabel]
    val envToLabelsMap = mutable.Map.empty[String, mutable.Set[EnvironmentLabel]]

    val cachedEnvMapper: RowMapper[Environment] = (rs, rowNum) => {
      val envId = rs.getCiId(ColumnAliases.Environments.ID)
      val env = envMap.getOrElseUpdate(envId, environmentMapper(cachedStageMapper(stageMap)).mapRow(rs, rowNum))

      if (rs.getCiId(ColumnAliases.EnvLabels.ID) != null) {
        envToLabelsMap.getOrElseUpdate(envId, mutable.Set.empty) += cachedLabelMapper(labelMap).mapRow(rs, rowNum)
      }

      env
    }

    var rowNum = 0
    while (rs.next()) {
      cachedEnvMapper.mapRow(rs, rowNum)
      rowNum += 1
    }

    envMap.view.map { case (envId, env) =>
      env.setLabels(envToLabelsMap.getOrElse(envId, Set.empty[EnvironmentLabel]).view.toList.asJava)
      env
    }.toSeq
  }

  def reservationResultSetExtractor: ResultSetExtractor[Option[EnvironmentReservation]] = rs => {
    if (rs.next()) {
      val labelMap = mutable.Map.empty[String, EnvironmentLabel]
      val appMap = mutable.Map.empty[String, Application]
      val safeLabelMapper: RowMapper[EnvironmentLabel] = (rs, rowNum) => if (rs.getCiId(ColumnAliases.EnvLabels.ID) != null) {
        cachedLabelMapper(labelMap).mapRow(rs, rowNum)
      } else {
        null
      }

      val safeAppMapper: RowMapper[Application] = (rs, rowNum) => if (rs.getCiId(ColumnAliases.Applications.ID) != null) {
        cachedApplicationMapper(appMap).mapRow(rs, rowNum)
      } else {
        null
      }

      var rowNum = 0
      val reservation = environmentReservationMapper().mapRow(rs, rowNum)
      if (rs.getCiId(ColumnAliases.Applications.ID) != null || rs.getCiId(ColumnAliases.EnvLabels.ID) != null) {
        safeLabelMapper.mapRow(rs, rowNum)
        safeAppMapper.mapRow(rs, rowNum)
        rowNum += 1
        while (rs.next()) {
          safeLabelMapper.mapRow(rs, rowNum)
          safeAppMapper.mapRow(rs, rowNum)
          rowNum += 1
        }
      }

      reservation.getEnvironment.setLabels(labelMap.valuesIterator.toSeq.asJava)
      reservation.setApplications(appMap.valuesIterator.toSeq.asJava)
      Some(reservation)
    } else {
      None
    }
  }

  val reservationSearchResultSetExtractor: ResultSetExtractor[Map[Environment, Seq[EnvironmentReservation]]] = rs => {
    // Search row: | Environment | Stage | Label | Reservation | Application |
    // [
    //   Reservation -> Environment -> Stage
    //                              -> [Label]
    //               -> [Application]
    // ]
    val envToEnvResMapMap = mutable.Map.empty[Environment, mutable.Map[String, EnvironmentReservation]]
    val stageMap = mutable.Map.empty[String, EnvironmentStage]
    val labelMap = mutable.Map.empty[String, EnvironmentLabel]
    val envMap = mutable.Map.empty[String, Environment]
    val appMap = mutable.Map.empty[String, Application]
    val envToLabelsMap = mutable.Map.empty[String, mutable.Set[EnvironmentLabel]]
    val resToAppsMap = mutable.Map.empty[String, mutable.Set[Application]]

    val cachedEnvMapper: RowMapper[Environment] = (rs, rowNum) => {
      val envId = rs.getCiId(ColumnAliases.Environments.ID)
      val env = envMap.getOrElseUpdate(envId, environmentMapper(cachedStageMapper(stageMap)).mapRow(rs, rowNum))

      if (rs.getCiId(ColumnAliases.EnvLabels.ID) != null) {
        envToLabelsMap.getOrElseUpdate(envId, mutable.Set.empty) += cachedLabelMapper(labelMap).mapRow(rs, rowNum)
      }

      env
    }

    def cachedResMapper(resMap: mutable.Map[String, EnvironmentReservation]): RowMapper[EnvironmentReservation] = (rs, rowNum) => {
      val resId = rs.getCiId(ColumnAliases.EnvReservations.ID)
      val res = resMap.getOrElseUpdate(resId, environmentReservationMapper(cachedEnvMapper).mapRow(rs, rowNum))

      if (rs.getCiId(ColumnAliases.Applications.ID) != null) {
        resToAppsMap.getOrElseUpdate(resId, mutable.Set.empty) += cachedApplicationMapper(appMap).mapRow(rs, rowNum)
      }

      res
    }

    var rowNum = 0
    while (rs.next()) {
      val environment = cachedEnvMapper.mapRow(rs, rowNum)
      val resMap = envToEnvResMapMap.getOrElseUpdate(environment, mutable.Map.empty[String, EnvironmentReservation])
      if (rs.getCiId(ColumnAliases.EnvReservations.ID) != null) {
        cachedResMapper(resMap).mapRow(rs, rowNum)
      }

      rowNum += 1
    }

    // Row processing done, add stuff to lists and transform to appropriate type
    envToEnvResMapMap.view.map { case (env, resMap) =>
      env.setLabels(envToLabelsMap.getOrElse(env.getId, Set.empty[EnvironmentLabel]).view.toList.asJava)

      env -> resMap.view.map { case (rId, res) =>
        res.setApplications(resToAppsMap.getOrElse(rId, Set.empty[Application]).view.toList.asJava)
        res
      }.toSeq
    }.toMap
  }

  def environmentStageMapper: RowMapper[EnvironmentStage] = (rs, _) => {
    val stage = new EnvironmentStage
    stage.setId(rs.getCiId(ColumnAliases.EnvStages.ID))
    stage.setTitle(rs.getString(ColumnAliases.EnvStages.TITLE))
    stage
  }

  def environmentLabelMapper: RowMapper[EnvironmentLabel] = (rs, _) => {
    val label = new EnvironmentLabel
    label.setId(rs.getCiId(ColumnAliases.EnvLabels.ID))
    label.setTitle(rs.getString(ColumnAliases.EnvLabels.TITLE))
    label.setColor(rs.getString(ColumnAliases.EnvLabels.COLOR))
    label
  }

  private def applicationMapper: RowMapper[Application] = (rs, _) => {
    // partial mapper, environments are missing
    val app = new Application
    app.setId(rs.getCiId(ColumnAliases.Applications.ID))
    app.setTitle(rs.getString(ColumnAliases.Applications.TITLE))
    app
  }

  private def environmentMapper(stageMapper: RowMapper[EnvironmentStage] = environmentStageMapper
                               ): RowMapper[Environment] = (rs, rowNum) => {
    // partial mapper, labels are missing
    val env = new Environment
    env.setId(rs.getCiId(ColumnAliases.Environments.ID))
    env.setTitle(rs.getString(ColumnAliases.Environments.TITLE))
    env.setDescription(rs.getString(ColumnAliases.Environments.DESCRIPTION))
    env.setStage(stageMapper.mapRow(rs, rowNum))
    env
  }

  private def environmentReservationMapper(envMapper: RowMapper[Environment] = environmentMapper(environmentStageMapper)
                                          ): RowMapper[EnvironmentReservation] = (rs, rowNum) => {
    // partial mapper for many-to-many, application is missing
    val reservation = new EnvironmentReservation
    reservation.setId(rs.getCiId(ColumnAliases.EnvReservations.ID))
    reservation.setStartDate(new Date(
      rs.getTimestamp(ColumnAliases.EnvReservations.START_DATE).getTime
    ))
    reservation.setEndDate(new Date(
      rs.getTimestamp(ColumnAliases.EnvReservations.END_DATE).getTime
    ))
    reservation.setNote(rs.getString(ColumnAliases.EnvReservations.NOTE))
    reservation.setEnvironment(envMapper.mapRow(rs, rowNum))
    reservation
  }

  private def cachedStageMapper(stageMap: mutable.Map[String, EnvironmentStage]): RowMapper[EnvironmentStage] = (rs, rowNum) => {
    val stageId = rs.getString(ColumnAliases.EnvStages.TITLE)
    stageMap.getOrElseUpdate(stageId, environmentStageMapper.mapRow(rs, rowNum))
  }

  private def cachedLabelMapper(labelMap: mutable.Map[String, EnvironmentLabel]): RowMapper[EnvironmentLabel] = (rs, rowNum) => {
    val labelId = rs.getCiId(ColumnAliases.EnvLabels.ID)
    labelMap.getOrElseUpdate(labelId, environmentLabelMapper.mapRow(rs, rowNum))
  }

  private def cachedApplicationMapper(appMap: mutable.Map[String, Application]): RowMapper[Application] = (rs, rowNum) => {
    val appId = rs.getCiId(ColumnAliases.Applications.ID)
    appMap.getOrElseUpdate(appId, applicationMapper.mapRow(rs, rowNum))
  }
}
