package com.xebialabs.xlrelease.environments.repository.sql.persistence.builder

import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.{Sql, SqlBuilder, SqlWithParameters}
import com.xebialabs.xlrelease.environments.repository.sql.persistence.schema.ApplicationSchema.{APPLICATIONS, APPLICATION_TO_ENVIRONMENTS}
import com.xebialabs.xlrelease.environments.repository.sql.persistence.schema.EnvironmentSchema.ENVIRONMENTS
import com.xebialabs.xlrelease.environments.repository.sql.persistence.toPersistedId
import com.xebialabs.xlrelease.repository.sql.persistence.CiId.CiId

object ApplicationSqlBuilder {
  def apply(aliasPrefix: String = "")(implicit dialect: Dialect): ApplicationSqlBuilder = {
    new ApplicationSqlBuilder(aliasPrefix)
  }
}

class ApplicationSqlBuilder(aliasPrefix: String)(implicit dialect: Dialect) extends SqlBuilder[ApplicationSqlBuilder] {
  private val baseAppAlias = makeAlias(aliasPrefix, "baseApp")
  private val appToEnvAlias = makeAlias(aliasPrefix, "appToEnv")
  private val envAlias = makeAlias(aliasPrefix, "env")

  private lazy val baseBuilder = new BaseApplicationSqlBuilder(baseAppAlias)

  // Applications with basic environment view without stages and labels for now
  override def build(): SqlWithParameters = {
    val (innerQuery, innerParams) = baseBuilder.select().build()
    val stmt =
      s"""
         |SELECT
         | $baseAppAlias.${ColumnAliases.Applications.ID} ${ColumnAliases.Applications.ID}
         | ,$baseAppAlias.${ColumnAliases.Applications.TITLE} ${ColumnAliases.Applications.TITLE}
         | ,$envAlias.${ENVIRONMENTS.ID} ${ColumnAliases.Environments.ID}
         | ,$envAlias.${ENVIRONMENTS.TITLE} ${ColumnAliases.Environments.TITLE}
         | ,$envAlias.${ENVIRONMENTS.DESCRIPTION} ${ColumnAliases.Environments.DESCRIPTION}
         | FROM (${indent(innerQuery)}) $baseAppAlias
         | LEFT JOIN ${APPLICATION_TO_ENVIRONMENTS.TABLE} $appToEnvAlias
         |  ON $baseAppAlias.${ColumnAliases.Applications.CI_UID} = $appToEnvAlias.${APPLICATION_TO_ENVIRONMENTS.APPLICATION_UID}
         | LEFT JOIN ${ENVIRONMENTS.TABLE} $envAlias
         |  ON $appToEnvAlias.${APPLICATION_TO_ENVIRONMENTS.ENVIRONMENT_UID} = $envAlias.${ENVIRONMENTS.CI_UID}
       """.stripMargin
    super.select(stmt)
    super.orderBy(ColumnAliases.Applications.ID)
    val fullSql = super.build()
    fullSql._1 -> (innerParams ++ fullSql._2)
  }

  def select(): ApplicationSqlBuilder = this

  def withTitleLike(title: String): ApplicationSqlBuilder = {
    baseBuilder.withTitleLike(title)
    this
  }

  def withTitle(title: String): ApplicationSqlBuilder = {
    baseBuilder.withTitle(title)
    this
  }

  def withEnvironmentTitles(titles: Seq[String]): ApplicationSqlBuilder = {
    baseBuilder.withEnvironmentTitles(titles)
    this
  }

  def withApplicationId(id: CiId): ApplicationSqlBuilder = {
    baseBuilder.withApplicationId(id)
    this
  }

  def withEnvironmentIdOrNull(environmentId: CiId): ApplicationSqlBuilder = {
    baseBuilder.withEnvironmentIdOrNull(environmentId)
    this
  }

  override def limitAndOffset(limit: Long, offset: Long): ApplicationSqlBuilder = {
    baseBuilder.limitAndOffset(limit, offset)
    this
  }

  override def orderBy(column: String): ApplicationSqlBuilder = {
    baseBuilder.orderBy(column)
    this
  }

  override def newInstance: ApplicationSqlBuilder = new ApplicationSqlBuilder(aliasPrefix)

  class BaseApplicationSqlBuilder(aliasPrefix: String)(implicit dialect: Dialect) extends SqlBuilder[BaseApplicationSqlBuilder] {
    private val appAlias = makeAlias(aliasPrefix, "app")
    private val appToEnvAlias = makeAlias(aliasPrefix, "appToEnv")
    private val envAlias = makeAlias(aliasPrefix, "env")

    def select(): BaseApplicationSqlBuilder = this

    // Many-to-many joins are not returned, they are only used for filtering
    // This is to make filtering work properly
    override def build(): SqlWithParameters = {
      val stmt =
        s"""
           |SELECT DISTINCT
           | $appAlias.${APPLICATIONS.CI_UID} ${ColumnAliases.Applications.CI_UID}
           | ,$appAlias.${APPLICATIONS.ID} ${ColumnAliases.Applications.ID}
           | ,$appAlias.${APPLICATIONS.TITLE} ${ColumnAliases.Applications.TITLE}
           | FROM ${APPLICATIONS.TABLE} $appAlias
           | LEFT JOIN ${APPLICATION_TO_ENVIRONMENTS.TABLE} $appToEnvAlias
           |  ON $appAlias.${APPLICATIONS.CI_UID} = $appToEnvAlias.${APPLICATION_TO_ENVIRONMENTS.APPLICATION_UID}
           | LEFT JOIN ${ENVIRONMENTS.TABLE} $envAlias
           |  ON $appToEnvAlias.${APPLICATION_TO_ENVIRONMENTS.ENVIRONMENT_UID} = $envAlias.${ENVIRONMENTS.CI_UID}
        """.stripMargin
      super.select(stmt)

      super.build()
    }

    def withTitleLike(title: String): BaseApplicationSqlBuilder = {
      like(s"$appAlias.${APPLICATIONS.TITLE}", title)
    }

    def withTitle(title: String): BaseApplicationSqlBuilder = {
      conditions += Sql(s"LOWER($appAlias.${APPLICATIONS.TITLE}) = LOWER(?)", Seq(title))
      this
    }

    def withEnvironmentTitles(titles: Seq[String]): BaseApplicationSqlBuilder = {
      conditions ++= whereInCondition(s"$envAlias.${ENVIRONMENTS.TITLE}", titles).toList
      this
    }

    def withApplicationId(id: CiId): BaseApplicationSqlBuilder = {
      conditions += Sql(s"$appAlias.${APPLICATIONS.ID} = ?", Seq(id))
      this
    }

    def withEnvironmentIdOrNull(environmentId: CiId): BaseApplicationSqlBuilder = {
      conditions += Sql(s"($envAlias.${ENVIRONMENTS.ID} = ? OR $envAlias.${ENVIRONMENTS.ID} IS NULL)", Seq(toPersistedId(environmentId)))
      this
    }

    override def newInstance: BaseApplicationSqlBuilder = new BaseApplicationSqlBuilder(aliasPrefix)

  }
}
