package com.xebialabs.xlrelease.environments.repository.sql.persistence

import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional
import com.xebialabs.xlrelease.db.sql.{LimitOffset, SqlWithParameters}
import com.xebialabs.xlrelease.domain.environments.EnvironmentReservation
import com.xebialabs.xlrelease.environments.repository.sql.persistence.builder.ColumnAliases
import com.xebialabs.xlrelease.environments.repository.sql.persistence.data.EnvironmentReservationRow
import com.xebialabs.xlrelease.environments.repository.sql.persistence.data.EnvironmentReservationRow.{reservationResultSetExtractor, reservationSearchResultSetExtractor}
import com.xebialabs.xlrelease.environments.repository.sql.persistence.schema.ApplicationSchema.APPLICATIONS
import com.xebialabs.xlrelease.environments.repository.sql.persistence.schema.EnvironmentLabelSchema.ENV_LABELS
import com.xebialabs.xlrelease.environments.repository.sql.persistence.schema.EnvironmentReservationSchema
import com.xebialabs.xlrelease.environments.repository.sql.persistence.schema.EnvironmentReservationSchema.{ENV_RES_TO_APP, ENV_RESERVATIONS => ER}
import com.xebialabs.xlrelease.environments.repository.sql.persistence.schema.EnvironmentSchema.{ENVIRONMENTS, ENV_TO_LABEL}
import com.xebialabs.xlrelease.environments.repository.sql.persistence.schema.EnvironmentStageSchema.ENV_STAGES
import com.xebialabs.xlrelease.repository.sql.persistence.CiId.CiId
import com.xebialabs.xlrelease.repository.sql.persistence.Utils.{params, _}
import com.xebialabs.xlrelease.repository.sql.persistence.{CiUid, PersistenceSupport}
import com.xebialabs.xlrelease.service.CiIdService
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.JdbcTemplate
import org.springframework.stereotype.Repository
import org.springframework.util.CollectionUtils

import java.util.Date
import scala.jdk.CollectionConverters._
import scala.util.Try

@Repository
@IsTransactional
class EnvironmentReservationPersistence @Autowired()(@Qualifier("xlrRepositoryJdbcTemplate") implicit val jdbcTemplate: JdbcTemplate,
                                                     @Qualifier("xlrRepositorySqlDialect") implicit val dialect: Dialect,
                                                     implicit val ciIdService: CiIdService,
                                                     val environmentPersistence: EnvironmentPersistence,
                                                     val applicationPersistence: ApplicationPersistence)
  extends PersistenceSupport with LimitOffset {

  def insert(reservation: EnvironmentReservation): CiId = {
    sanitizeReservationInput(reservation)

    val environmentUid = environmentPersistence.findUidById(reservation.getEnvironment.getId)
      .getOrElse(throw new NotFoundException(s"Environment [${reservation.getEnvironment.getId}] not found"))

    val applicationIdsToUids = getApplicationUids(reservation)

    val insertReservationStmt =
      s"""INSERT INTO ${ER.TABLE} (
         |${ER.ID},
         |${ER.ENVIRONMENT_UID},
         |${ER.START_DATE},
         |${ER.END_DATE},
         |${ER.NOTE})
         |VALUES (
         |:${ER.ID},
         |:${ER.ENVIRONMENT_UID},
         |:${ER.START_DATE},
         |:${ER.END_DATE},
         |:${ER.NOTE})
       """.stripMargin
    val resId = createPersistedId[EnvironmentReservation]
    sqlInsert(pkName(ER.CI_UID),
      insertReservationStmt,
      params(ER.ID -> resId,
        ER.ENVIRONMENT_UID -> environmentUid,
        ER.START_DATE -> reservation.getStartDate,
        ER.END_DATE -> reservation.getEndDate,
        ER.NOTE -> reservation.getNote),
      (ciUid: CiUid) => insertEnvResToAppReferences(ciUid, applicationIdsToUids.valuesIterator.toSet)
    )
    toDisplayId(resId)
  }

  private val SELECT_RESERVATION_BY_RESERVATION_ID =
    s"""
       |SELECT
       | res.${ER.CI_UID} ${ColumnAliases.EnvReservations.CI_UID},
       | res.${ER.ID} ${ColumnAliases.EnvReservations.ID},
       | res.${ER.START_DATE} ${ColumnAliases.EnvReservations.START_DATE},
       | res.${ER.END_DATE} ${ColumnAliases.EnvReservations.END_DATE},
       | res.${ER.NOTE} ${ColumnAliases.EnvReservations.NOTE},
       | env.${ENVIRONMENTS.CI_UID} ${ColumnAliases.Environments.CI_UID},
       | env.${ENVIRONMENTS.ID} ${ColumnAliases.Environments.ID},
       | envLabel.${ENV_LABELS.CI_UID} ${ColumnAliases.EnvLabels.CI_UID},
       | envLabel.${ENV_LABELS.ID} ${ColumnAliases.EnvLabels.ID},
       | envLabel.${ENV_LABELS.TITLE} ${ColumnAliases.EnvLabels.TITLE},
       | envLabel.${ENV_LABELS.COLOR} ${ColumnAliases.EnvLabels.COLOR},
       | envStages.${ENV_STAGES.CI_UID} ${ColumnAliases.EnvStages.CI_UID},
       | envStages.${ENV_STAGES.ID} ${ColumnAliases.EnvStages.ID},
       | envStages.${ENV_STAGES.TITLE} ${ColumnAliases.EnvStages.TITLE},
       | env.${ENVIRONMENTS.CONTENT} ${ColumnAliases.Environments.CONTENT},
       | app.${APPLICATIONS.CI_UID} ${ColumnAliases.Applications.CI_UID},
       | app.${APPLICATIONS.ID} ${ColumnAliases.Applications.ID},
       | app.${APPLICATIONS.CONTENT} ${ColumnAliases.Applications.CONTENT}
       | FROM ${ER.TABLE} res
       | LEFT JOIN ${ENV_RES_TO_APP.TABLE} resToApp
       |  ON res.${ER.CI_UID} = resToApp.${ENV_RES_TO_APP.RESERVATION_UID}
       | LEFT JOIN ${APPLICATIONS.TABLE} app
       |  ON resToApp.${ENV_RES_TO_APP.APPLICATION_UID} = app.${APPLICATIONS.CI_UID}
       | LEFT JOIN ${ENVIRONMENTS.TABLE} env
       |  ON res.${ER.ENVIRONMENT_UID} = env.${ENVIRONMENTS.CI_UID}
       | LEFT JOIN ${ENV_TO_LABEL.TABLE} envToLabel
       |  ON envToLabel.${ENV_TO_LABEL.ENVIRONMENT_UID} = env.${ENVIRONMENTS.CI_UID}
       | LEFT JOIN ${ENV_LABELS.TABLE} envLabel
       |  ON envToLabel.${ENV_TO_LABEL.LABEL_UID} = envLabel.${ENV_LABELS.CI_UID}
       | LEFT JOIN ${ENV_STAGES.TABLE} envStages
       |  ON env.${ENVIRONMENTS.ENV_STAGE_UID} = envStages.${ENV_STAGES.CI_UID}
       | WHERE res.${ER.ID} = :${ColumnAliases.EnvReservations.ID}
        """.stripMargin

  def findById(environmentReservationId: CiId): Option[EnvironmentReservationRow] =
    sqlQuery(SELECT_RESERVATION_BY_RESERVATION_ID,
      params(ColumnAliases.EnvReservations.ID -> toPersistedId(environmentReservationId)),
      reservationResultSetExtractor
    )

  def findUidById(reservationId: CiId): Option[CiUid] = {
    val stmt =
      s"""|SELECT ${ER.CI_UID}
          |FROM ${ER.TABLE}
          |WHERE ${ER.ID} = :${ER.ID}""".stripMargin
    sqlQuery(stmt, params(ER.ID -> toPersistedId(reservationId)), rs => CiUid(rs.getInt(ER.CI_UID))).headOption
  }

  def update(reservation: EnvironmentReservation): Boolean = {
    val reservationUid = findUidById(reservation.getId)
      .getOrElse(throw new NotFoundException(s"Reservation [${reservation.getId}] not found"))
    val environmentUid = environmentPersistence.findUidById(reservation.getEnvironment.getId)
      .getOrElse(throw new NotFoundException(s"Environment [${reservation.getEnvironment.getId}] not found"))

    sanitizeReservationInput(reservation)

    updateEnvResToAppReferences(reservation, reservationUid)

    val updateReservationStmt =
      s"""
         |UPDATE ${ER.TABLE}
         | SET
         |  ${ER.START_DATE} = :${ER.START_DATE},
         |  ${ER.END_DATE} = :${ER.END_DATE},
         |  ${ER.NOTE} = :${ER.NOTE},
         |  ${ER.ENVIRONMENT_UID} = :${ER.ENVIRONMENT_UID}
         | WHERE
         |  ${ER.CI_UID} = :${ER.CI_UID}
         |""".stripMargin
    sqlUpdate(updateReservationStmt,
      params(
        ER.START_DATE -> reservation.getStartDate,
        ER.END_DATE -> reservation.getEndDate,
        ER.NOTE -> reservation.getNote,
        ER.ENVIRONMENT_UID -> environmentUid,
        ER.CI_UID -> reservationUid
      ),
      _ == 1
    )
  }

  def search(sqlWithParameters: SqlWithParameters): Map[CiId, Seq[EnvironmentReservationRow]] = {
    val (sql, params) = sqlWithParameters
    jdbcTemplate.query(
      sql,
      reservationSearchResultSetExtractor,
      params: _*
    )
  }

  def delete(environmentReservationId: CiId): Try[Boolean] = {
    val stmt =
      s"""
         |DELETE FROM ${ER.TABLE}
         | WHERE ${ER.ID} = :reservationId
       """.stripMargin
    sqlExec(stmt, params("reservationId" -> toPersistedId(environmentReservationId)), ps => Try(ps.execute()))
  }

  def exists(environmentId: String, applicationId: String, date: Date): Boolean = {
    val stmt =
      s"""
         |SELECT 1
         |FROM ${ER.TABLE}
         |WHERE ${ER.START_DATE} <= :dateToSearch
         | AND ${ER.END_DATE} >= :dateToSearch
         | AND ${ER.ENVIRONMENT_UID} = (
         |  SELECT ${ENVIRONMENTS.CI_UID}
         |  FROM ${ENVIRONMENTS.TABLE}
         |  WHERE ${ENVIRONMENTS.ID} = :environmentId
         | )
         | AND ${ER.CI_UID} IN (
         |  SELECT ${ENV_RES_TO_APP.RESERVATION_UID}
         |  FROM ${ENV_RES_TO_APP.TABLE}
         |  WHERE ${ENV_RES_TO_APP.APPLICATION_UID} = (
         |   SELECT ${APPLICATIONS.CI_UID}
         |   FROM ${APPLICATIONS.TABLE}
         |   WHERE ${APPLICATIONS.ID} = :applicationId
         |  )
         | )
     """.stripMargin
    findOne {
      sqlQuery(stmt,
        params("environmentId" -> toPersistedId(environmentId), "applicationId" -> toPersistedId(applicationId), "dateToSearch" -> date),
        _.getInt(1) == 1)
    }.getOrElse(false)
  }

  def findNearestComing(environmentId: String, applicationId: String, date: Date): Option[Date] = {
    val stmt =
      s"""
         |SELECT ${ER.START_DATE}
         |FROM ${ER.TABLE}
         |WHERE ${ER.END_DATE} >= :dateToSearch
         | AND ${ER.ENVIRONMENT_UID} = (
         |  SELECT ${ENVIRONMENTS.CI_UID}
         |  FROM ${ENVIRONMENTS.TABLE}
         |  WHERE ${ENVIRONMENTS.ID} = :environmentId
         | )
         | AND ${ER.CI_UID} IN (
         |  SELECT ${ENV_RES_TO_APP.RESERVATION_UID}
         |  FROM ${ENV_RES_TO_APP.TABLE}
         |  WHERE ${ENV_RES_TO_APP.APPLICATION_UID} = (
         |   SELECT ${APPLICATIONS.CI_UID}
         |   FROM ${APPLICATIONS.TABLE}
         |   WHERE ${APPLICATIONS.ID} = :applicationId
         |  )
         | )
         |ORDER BY ${ER.START_DATE} ASC
     """.stripMargin
    findOne {
      sqlQuery(addLimitAndOffset(stmt, Option(1)),
        params("environmentId" -> toPersistedId(environmentId), "applicationId" -> toPersistedId(applicationId), "dateToSearch" -> date),
        r => Option(r.getTimestamp(1)))
    }.flatten
  }

  private def getApplicationUids(reservation: EnvironmentReservation): Map[CiId, CiUid] = {
    if (CollectionUtils.isEmpty(reservation.getApplications)) {
      Map.empty[CiId, CiUid]
    } else {
      val applicationIds = reservation.getApplications.asScala.map(_.getId).toSet
      val applicationIdsToUids = applicationPersistence.getUidsByIds(applicationIds)
      val applicationDiff = applicationIds.map(toDisplayId).diff(applicationIdsToUids.keySet)
      if (applicationDiff.nonEmpty) {
        throw new NotFoundException(s"Applications [${applicationDiff.mkString(", ")}] not found")
      }
      applicationIdsToUids
    }
  }

  private def insertEnvResToAppReferences(reservationUid: CiUid, appUids: Set[CiUid]) = {
    val insertEnvResToAppStmt =
      s"""INSERT INTO ${ENV_RES_TO_APP.TABLE} (
         |${ENV_RES_TO_APP.RESERVATION_UID},
         |${ENV_RES_TO_APP.APPLICATION_UID}
         |)
         |VALUES (
         |:${ENV_RES_TO_APP.RESERVATION_UID},
         |:${ENV_RES_TO_APP.APPLICATION_UID}
         |)
         |""".stripMargin

    sqlBatch(insertEnvResToAppStmt,
      appUids.map { appUid =>
        params(
          ENV_RES_TO_APP.RESERVATION_UID -> reservationUid,
          ENV_RES_TO_APP.APPLICATION_UID -> appUid
        )
      }
    )
  }

  private def updateEnvResToAppReferences(reservation: EnvironmentReservation, reservationUid: CiUid) = {
    val applicationIdsToUids = getApplicationUids(reservation)

    val deleteEnvResToAppRefsStmt =
      s"""
         |DELETE FROM ${ENV_RES_TO_APP.TABLE}
         | WHERE
         | ${ENV_RES_TO_APP.RESERVATION_UID} = :${ENV_RES_TO_APP.RESERVATION_UID}
       """.stripMargin
    sqlExec(deleteEnvResToAppRefsStmt, params(ENV_RES_TO_APP.RESERVATION_UID -> reservationUid), _.execute())
    insertEnvResToAppReferences(reservationUid, applicationIdsToUids.valuesIterator.toSet)
  }

  private def sanitizeReservationInput(reservation: EnvironmentReservation): Unit = {
    reservation.setNote(reservation.getNote.trimAndTruncate(EnvironmentReservationSchema.NOTE_LENGTH))
  }
}
