package com.xebialabs.xlrelease.environments.initialize

import com.xebialabs.deployit.server.api.upgrade.{Upgrade, Version}
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.domain.environments.{Application, Environment}
import com.xebialabs.xlrelease.environments.repository.sql.persistence.builder.ColumnAliases
import com.xebialabs.xlrelease.environments.repository.sql.persistence.schema.ApplicationSchema.{APPLICATIONS, APPLICATION_TO_ENVIRONMENTS}
import com.xebialabs.xlrelease.environments.repository.sql.persistence.schema.EnvironmentSchema.ENVIRONMENTS
import com.xebialabs.xlrelease.repository.sql.persistence.Utils.params
import com.xebialabs.xlrelease.repository.sql.persistence.{CiUid, PersistenceSupport}
import com.xebialabs.xlrelease.serialization.json.utils.CiSerializerHelper.serialize
import com.xebialabs.xlrelease.upgrade.Components.XL_RELEASE_COMPONENT
import com.xebialabs.xlrelease.upgrade.UpgradeSupport.{BatchSupport, ParallelSupport, TransactionSupport}
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.jdbc.core.{JdbcTemplate, ResultSetExtractor}
import org.springframework.stereotype.Component
import org.springframework.transaction.support.TransactionTemplate

import scala.collection.mutable
import scala.jdk.CollectionConverters._


@Component
class XlRelease243ApplicationAddJsonContentUpgrader(@Qualifier("xlrRepositoryJdbcTemplate") val jdbcTemplate: JdbcTemplate,
                                                    @Qualifier("xlrRepositorySqlDialect") implicit val dialect: Dialect,
                                                    @Qualifier("xlrRepositoryTransactionTemplate") val transactionTemplate: TransactionTemplate)
  extends Upgrade with BatchSupport with ParallelSupport with PersistenceSupport with TransactionSupport with Logging {

  override def upgradeVersion(): Version = Version.valueOf(XL_RELEASE_COMPONENT, "24.3.0#1")

  override def doUpgrade(): Boolean = {
    logger.info("Updating Applications with CI JSON content")
    val applications = findApplicationRows
    doInBatch(applications) { batch =>
      doInTransaction {
        doInParallel(batch.items) {
          case (applicationUid, application) =>
            updateApplicationContent(applicationUid, application)
        }
      }
    }
    logger.info("Finished updating Applications with CI JSON content")
    true
  }

  private val STMT_UPDATE_APPLICATIONS =
    s"""|UPDATE ${APPLICATIONS.TABLE}
        | SET
        |  ${APPLICATIONS.CONTENT} = :${APPLICATIONS.CONTENT}
        | WHERE
        |  ${APPLICATIONS.CI_UID} = :${APPLICATIONS.CI_UID}
       """.stripMargin

  private def updateApplicationContent(applicationUid: CiUid, application: Application): Unit = {
    sqlExecWithContent(
      STMT_UPDATE_APPLICATIONS,
      params(
        APPLICATIONS.CI_UID -> applicationUid,
      ),
      APPLICATIONS.CONTENT -> serialize(application),
      checkCiUpdated(application.getId)
    )
  }

  private val STMT_FIND_APPS =
    s"""
       |SELECT
       |  ${APPLICATIONS.CI_UID},
       |  ${APPLICATIONS.ID},
       |  ${APPLICATIONS.TITLE}
       |FROM
       |  ${APPLICATIONS.TABLE}
     """.stripMargin

  private def findApplicationRows: Vector[(CiUid, Application)] = {
    sqlQuery(STMT_FIND_APPS, params(), applicationMapper())
  }

  private def applicationMapper(): ResultSetExtractor[Vector[(CiUid, Application)]] = rs => {
    val apps = mutable.Set.empty[(CiUid, Application)]

    while(rs.next()) {
      val app = new Application
      app.setTitle(rs.getString(APPLICATIONS.TITLE))
      app.setId(rs.getString(APPLICATIONS.ID))
      val ciuid = rs.getInt(APPLICATIONS.CI_UID)
      val envs = findEnvironmentsForApplication(ciuid).toList.asJava
      app.setEnvironments(envs)
      apps.add((ciuid, app))
    }
    apps.toVector
  }



  private val STMT_FIND_ENVS_FOR_APPS =
    s"""
       |SELECT
       |  env.${ENVIRONMENTS.CI_UID} ${ColumnAliases.Environments.CI_UID},
       |  env.${ENVIRONMENTS.ID} ${ColumnAliases.Environments.ID},
       |  env.${ENVIRONMENTS.TITLE} ${ColumnAliases.Environments.TITLE},
       |  env.${ENVIRONMENTS.DESCRIPTION} ${ColumnAliases.Environments.DESCRIPTION}
       |FROM
       |  ${APPLICATIONS.TABLE} app
       |LEFT JOIN ${APPLICATION_TO_ENVIRONMENTS.TABLE} appenv
       |  on app.${APPLICATIONS.CI_UID} = appenv.${APPLICATION_TO_ENVIRONMENTS.APPLICATION_UID}
       |LEFT JOIN ${ENVIRONMENTS.TABLE} env
       |  on appenv.${APPLICATION_TO_ENVIRONMENTS.ENVIRONMENT_UID} = env.${ENVIRONMENTS.CI_UID}
       |WHERE app.${APPLICATIONS.CI_UID} = :appUid
     """.stripMargin

  // needs only the reference of environment with ID
  private def findEnvironmentsForApplication(appUid: Int): Vector[Environment] =
    sqlQuery(STMT_FIND_ENVS_FOR_APPS, params("appUid" -> appUid), environmentMapper())

  private def environmentMapper(): ResultSetExtractor[Vector[Environment]] = rs => {
    val envs = mutable.Set.empty[Environment]
    while(rs.next()) {
      val env = new Environment
      env.setTitle(rs.getString(ColumnAliases.Environments.TITLE))
      env.setId(rs.getString(ColumnAliases.Environments.ID))
      env.setDescription(rs.getString(ColumnAliases.Environments.DESCRIPTION))
      envs.add(env)
    }
    envs.toVector
  }
}
