import { all, call, put, takeLatest, throttle } from 'redux-saga/effects';
import isEmpty from 'lodash/isEmpty';
import ToastrFactory from '@xlr-ui/app/js/util/toastrFactory';
import { httpDELETE, httpPOST, httpPUT } from '@xlr-ui/app/features/common/services/http';
import { stageState } from './stage.reducer';
import { actionTypes as schedulerActionTypes } from '../../../actions/scheduling-actions';
import { ScheduleInitHeader, Stage } from '../../../application-list.types';
import { PayloadAction } from '@reduxjs/toolkit';
import { byTitle } from '../../../utils';

export const toastr = ToastrFactory();

const { setStages, searchStages, deleteStage, saveStage } = stageState.actions;
export function* _searchStages(action: PayloadAction<string>) {
    try {
        const { data: stages } = yield call(httpPOST, 'api/v1/environments/stages/search', { title: action.payload });
        yield put(setStages(stages.sort(byTitle)));
    } catch (e) {
        // eslint-disable-next-line no-console,angular/log
        yield call(console.error, 'Stages saga: ', e);
    }
}

export function* _saveStage(action: PayloadAction<Stage>) {
    const stage = action.payload;
    try {
        if (isEmpty(stage.id)) {
            yield call(httpPOST, 'api/v1/environments/stages', stage);
            yield call(toastr.success, `Stage [${stage.title}] created`);
        } else {
            yield call(httpPUT, `api/v1/environments/stages/${stage.id}`, stage);
            yield call(toastr.success, `Stage [${stage.title}] updated`);
        }
        yield put(searchStages(''));
    } catch (e) {
        // eslint-disable-next-line no-console,angular/log
        yield call(console.error, 'Stages saga: ', e);
    }
}

export function* _deleteStage(action: PayloadAction<Stage>) {
    const stage = action.payload;

    try {
        yield call(httpDELETE, `api/v1/environments/stages/${stage.id}`);
        yield call(toastr.success, `Stage [${stage.title}] deleted`);
        yield put(searchStages(''));
    } catch (e) {
        // eslint-disable-next-line no-console,angular/log
        yield call(console.error, 'Stages saga: ', e);
    }
}

export function* _scheduleInitHeader(action: PayloadAction<ScheduleInitHeader>) {
    const stages = action.payload.stages;
    yield put(setStages(stages.sort(byTitle)));
}

const debounce = 200;
export default function* stageSaga() {
    yield all([
        takeLatest(saveStage, _saveStage),
        takeLatest(deleteStage, _deleteStage),
        takeLatest(searchStages, _searchStages),
        throttle(debounce, schedulerActionTypes.SCHEDULE_INIT_HEADER_SUCCESS, _scheduleInitHeader),
    ]);
}
