package com.xebialabs.xlrelease.environments.repository.sql

import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.xlrelease.api.v1.filter.{ApplicationEnvironmentFilterScope, EnvironmentFilters}
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.transaction.{IsReadOnly, IsTransactional}
import com.xebialabs.xlrelease.domain.environments.{Environment, EnvironmentLabel, EnvironmentStage}
import com.xebialabs.xlrelease.environments.repository.sql.persistence.EnvironmentPersistence
import com.xebialabs.xlrelease.environments.repository.sql.persistence.builder.{ColumnAliases, EnvironmentSqlBuilder}
import com.xebialabs.xlrelease.environments.repository.{EnvironmentLabelRepository, EnvironmentRepository, EnvironmentStageRepository}
import com.xebialabs.xlrelease.repository.Page
import com.xebialabs.xlrelease.repository.sql.SqlRepositoryAdapter
import com.xebialabs.xlrelease.repository.sql.persistence.CiId.CiId
import com.xebialabs.xlrelease.repository.sql.persistence.FolderPersistence
import com.xebialabs.xlrelease.service.CiIdService
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.stereotype.Repository

import scala.collection.mutable
import scala.jdk.CollectionConverters._

@IsTransactional
@Repository
class SqlEnvironmentRepository @Autowired()(@Qualifier("xlrRepositorySqlDialect") implicit val dialect: Dialect,
                                            implicit val ciIdService: CiIdService,
                                            implicit val repositoryAdapter: SqlRepositoryAdapter,
                                            implicit val environmentLabelRepository: EnvironmentLabelRepository,
                                            implicit val environmentStageRepository: EnvironmentStageRepository,
                                            folderPersistence: FolderPersistence,
                                            environmentPersistence: EnvironmentPersistence,
                                           )
  extends EnvironmentRepository with Logging {

  @IsReadOnly
  override def searchEnvironments(environmentFilters: EnvironmentFilters, page: Page): Seq[Environment] = {
    val query = EnvironmentSqlBuilder()
      .select()
      .orderBy(ColumnAliases.Environments.CI_UID)
      .limitAndOffset(page.resultsPerPage, page.offset)

    Option(environmentFilters).map { f =>
      Option(f.getTitle).foreach(query.withTitleLike)
      Option(f.getStage).foreach(query.withStageTitle)
      query.applyFolderFiltering(Option(f.getScope), Option(f.getFolderId).map(folderPersistence.getUid))
      Option(f.getLabels).foreach(label => query.withLabelTitles(label.asScala))
    }.getOrElse(query.applyFolderFiltering(Some(ApplicationEnvironmentFilterScope.GLOBAL), None))

    implicit val cachedStages: mutable.Map[CiId, EnvironmentStage] = mutable.Map.empty
    implicit val cachedLabels: mutable.Map[CiId, EnvironmentLabel] = mutable.Map.empty
    environmentPersistence.search(query.build()).map(mapEnvironmentContent)
  }

  @IsReadOnly
  override def findEnvironmentById(environmentId: String): Environment = {
    environmentPersistence.findById(environmentId)
      .map(mapEnvironmentContent)
      .getOrElse(throw new NotFoundException(s"Environment [$environmentId] not found"))
  }

  @IsReadOnly
  override def findEnvironmentByTitle(environmentTitle: String): Environment = {
    environmentPersistence.findByTitle(environmentTitle).map(mapEnvironmentContent)
      .getOrElse(throw new NotFoundException(s"Environment [$environmentTitle] not found"))
  }

  @IsReadOnly
  override def findEnvironmentByTitleAndFolder(title: String, maybeFolder: Option[String]): Option[Environment] =
    environmentPersistence.findByTitleAndFolder(title, maybeFolder).map(mapEnvironmentContent)

  @IsReadOnly
  def findInFolderByCorrelationId(folderId: String, environmentUid: String): Option[Environment] =
    environmentPersistence.findInFolderByCorrelationId(folderId, environmentUid).map(mapEnvironmentContent)

  override def createEnvironment(environment: Environment): Environment = {
    interceptCreate(environment)
    val envId = environmentPersistence.insert(environment)
    val createdEnv = findEnvironmentById(envId)
    createdEnv
  }

  override def updateEnvironment(environment: Environment): Environment = {
    environmentPersistence.update(environment)
    findEnvironmentById(environment.getId)
  }

  override def deleteEnvironment(environmentId: String): Unit = {
    environmentPersistence.delete(environmentId)
  }

  override def fetchEnvironments(environmentIds: List[CiId]): Seq[Environment] = {
    environmentPersistence.fetchEnvironments(environmentIds).map(mapEnvironmentContent)

  }

  override def fetchEnvironmentsByFolderId(folderId: String): Seq[Environment] = {
    environmentPersistence.fetchEnvironmentsByFolderId(folderId).map(mapEnvironmentContent)

  }

  @IsReadOnly
  override def tenantEnvironmentCount(tenantId: String): Int = {
    environmentPersistence.tenantEnvironmentCount(tenantId)
  }
}
