package com.xebialabs.xlrelease.environments.initialize

import com.xebialabs.deployit.server.api.upgrade.Version
import com.xebialabs.xlrelease.domain.environments.EnvironmentStage
import com.xebialabs.xlrelease.environments.repository.EnvironmentStageRepository
import com.xebialabs.xlrelease.upgrade.Components.XL_RELEASE_COMPONENT
import com.xebialabs.xlrelease.upgrade.common.{BaseInitializingUpgrade, ClassNameToString}
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

import scala.util.Try

object DefaultEnvironmentStagesInitializer {
  val DEFAULT_DEV = "Development"
  val DEFAULT_TEST = "Test"
  val DEFAULT_ACCEPT = "Acceptance"
  val DEFAULT_PROD = "Production"
}

@Component
class DefaultEnvironmentStagesInitializer @Autowired()(environmentStageRepository: EnvironmentStageRepository)
  extends BaseInitializingUpgrade
    with ClassNameToString
    with Logging {

  import DefaultEnvironmentStagesInitializer._

  override def upgradeVersion(): Version = Version.valueOf(XL_RELEASE_COMPONENT, "8.5.0#1")

  override def doUpgrade(): Boolean = {
    logger.info("Creating default environment stages: Development, Test, Acceptance, Production")
    val dev = EnvironmentStage.create(DEFAULT_DEV)
    val test = EnvironmentStage.create(DEFAULT_TEST)
    val acceptance = EnvironmentStage.create(DEFAULT_ACCEPT)
    val prod = EnvironmentStage.create(DEFAULT_PROD)
    try {
      if (!exists(DEFAULT_DEV)) {
        environmentStageRepository.create(dev)
      }
      if (!exists(DEFAULT_TEST)) {
        environmentStageRepository.create(test)
      }
      if (!exists(DEFAULT_ACCEPT)) {
        environmentStageRepository.create(acceptance)
      }
      if (!exists(DEFAULT_PROD)) {
        environmentStageRepository.create(prod)
      }
    } catch {
      case e =>
        logger.error("Error initializing Environment Stages: ", e)
        throw e
    }

    logger.debug("Finished creating default environment stages")
    true
  }

  private def exists(stageTitle: String): Boolean = {
    Try(environmentStageRepository.findByTitle(stageTitle)).fold(_ => false, _ => true)
  }
}
