package com.xebialabs.xlrelease.environments.repository.sql

import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.repository.ItemInUseException
import com.xebialabs.xlrelease.api.v1.filter.EnvironmentStageFilters
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.transaction.{IsReadOnly, IsTransactional}
import com.xebialabs.xlrelease.domain.environments.EnvironmentStage
import com.xebialabs.xlrelease.environments.repository.EnvironmentStageRepository
import com.xebialabs.xlrelease.environments.repository.sql.persistence.builder.EnvironmentStageSqlBuilder
import com.xebialabs.xlrelease.environments.repository.sql.persistence.data.EnvironmentStageRow
import com.xebialabs.xlrelease.environments.repository.sql.persistence.schema.EnvironmentStageSchema.ENV_STAGES
import com.xebialabs.xlrelease.environments.repository.sql.persistence.{EnvironmentStagePersistence, toDisplayId}
import com.xebialabs.xlrelease.repository.Page
import com.xebialabs.xlrelease.service.CiIdService
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.stereotype.Repository

@IsTransactional
@Repository
class SqlEnvironmentStageRepository @Autowired()(@Qualifier("xlrRepositorySqlDialect") implicit val dialect: Dialect,
                                                 implicit val ciIdService: CiIdService,
                                                 environmentStagePersistence: EnvironmentStagePersistence)
  extends EnvironmentStageRepository with Logging {

  @IsReadOnly
  override def search(filters: EnvironmentStageFilters, page: Page): Seq[EnvironmentStage] = {
    val query = new EnvironmentStageSqlBuilder()
      .select()
      .orderBy(ENV_STAGES.ID)
      .limitAndOffset(page.resultsPerPage, page.offset)

    if (filters != null && filters.getTitle != null) {
      query.withTitleLike(filters.getTitle)
    }

    environmentStagePersistence.search(query.build()).map(toEnvironmentStage)
  }

  @IsReadOnly
  override def findById(environmentStageId: String): EnvironmentStage = {
    environmentStagePersistence.findById(environmentStageId).map(toEnvironmentStage)
      .getOrElse(throw new NotFoundException(s"Environment stage [$environmentStageId] not found"))
  }

  @IsReadOnly
  override def findByTitle(environmentStageTitle: String): EnvironmentStage = {
    environmentStagePersistence.findByTitle(environmentStageTitle).map(toEnvironmentStage)
      .getOrElse(throw new NotFoundException(s"Environment stage [$environmentStageTitle] not found"))
  }

  override def create(environmentStage: EnvironmentStage): EnvironmentStage = {
    val id = environmentStagePersistence.insert(environmentStage)
    findById(id)
  }

  override def update(environmentStage: EnvironmentStage): EnvironmentStage = {
    environmentStagePersistence.update(environmentStage)
    findById(environmentStage.getId)
  }

  override def delete(environmentStageId: String): Unit = {
    val usedBy = environmentStagePersistence.usedBy(environmentStageId)
    if (usedBy.nonEmpty) {
      val (first10, rest) = usedBy.splitAt(10)
      logger.warn(s"""Cannot delete environment stage [$environmentStageId] because it's used
                     | by the following environments: [${usedBy.mkString(", ")}]
         """.stripMargin)
      throw new ItemInUseException(
        s"""Cannot delete environment stage [$environmentStageId] because it's used
           | by the following environments: [${first10.mkString(", ")}]
           | ${if (rest.nonEmpty) ", and more"}
         """.stripMargin)
    } else {
      environmentStagePersistence.delete(environmentStageId)
    }
  }

  def toEnvironmentStage(row: EnvironmentStageRow): EnvironmentStage = {
    val stage = new EnvironmentStage()
    stage.setId(toDisplayId(row.id))
    stage.setTitle(row.title)
    stage
  }
}
