import { all, call, put, select, takeLatest, throttle } from 'redux-saga/effects';
import { actionTypes as schedulerActionTypes } from '../../../actions/scheduling-actions';
import getAngularService from '@xlr-ui/app/features/common/services/angular-accessor';
import isEmpty from 'lodash/isEmpty';
import { applicationLists } from './application-list.reducer';
import { applicationForm } from './application-form.reducer';
import { PayloadAction } from '@reduxjs/toolkit';
import ToastrFactory from '@xlr-ui/app/js/util/toastrFactory';
import { Application, ApplicationEdit, ApplicationName, ScheduleInitHeader } from '../../../application-list.types';
import { httpDELETE, httpPOST } from '@xlr-ui/app/features/common/services/http';
import { byTitle } from '../../../utils';
import { getApplicationsSelector } from './application-list.selectors';
import { IStateService } from 'angular-ui-router';

export const toastr = ToastrFactory();

const { searchApplications, setApplications, deleteApplication, editApplication, createApplication } = applicationLists.actions;
const { onFormClean } = applicationForm.actions;
export function* searchApplicationsAction(action: { payload: ApplicationName }) {
    try {
        const { data: applications } = yield call(httpPOST, 'api/v1/applications/search', { folderId: action.payload.folderId, title: action.payload.title });
        yield put(setApplications(applications));
    } catch (e) {
        // eslint-disable-next-line angular/log
        yield call(console.error, 'Application list saga: ', e);
    }
}

export function* goToApplicationCreate(action: PayloadAction<string>) {
    yield put(onFormClean());
    const $state: IStateService = yield call(getAngularService, '$state');
    if (isEmpty(action.payload)) {
        $state.go('applicationCreate');
    } else {
        $state.go('folders.detail.applicationCreate');
    }
}

export function* goToApplicationEdit(action: PayloadAction<ApplicationEdit>) {
    const $state: IStateService = yield call(getAngularService, '$state');
    if (isEmpty(action.payload.folderId)) {
        $state.go('applicationCreate', { applicationId: action.payload.appId });
    } else {
        $state.go('folders.detail.applicationCreate', { applicationId: action.payload.appId });
    }
}

export function* deleteApplicationAction(action: PayloadAction<Application>) {
    try {
        const application = action.payload;
        yield call(httpDELETE, `api/v1/applications/${application.id}`);
        const apps: Array<Application> = yield select(getApplicationsSelector);
        yield put(setApplications(apps.filter((app) => app.id !== application.id)));
        yield call(toastr.success, `Deleted application [${application.title}]`);
    } catch (e) {
        // eslint-disable-next-line angular/log
        yield call(console.error, 'Application list saga: ', e);
    }
}

export function* scheduleInitHeader(action: PayloadAction<ScheduleInitHeader>) {
    const apps = action.payload.applications.sort(byTitle);
    yield put(setApplications(apps));
}

const debounce = 200;
export default function* applicationListSaga() {
    yield all([
        takeLatest(createApplication, goToApplicationCreate),
        takeLatest(editApplication, goToApplicationEdit),
        takeLatest(deleteApplication, deleteApplicationAction),
        throttle(debounce, searchApplications, searchApplicationsAction),
        throttle(debounce, schedulerActionTypes.SCHEDULE_INIT_HEADER_SUCCESS, scheduleInitHeader),
    ]);
}
