package com.xebialabs.xlrelease.externalproviders.conjur;

import java.net.URI;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import javax.ws.rs.client.Client;
import javax.ws.rs.client.ClientBuilder;
import javax.ws.rs.client.WebTarget;
import javax.ws.rs.core.Response;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.cyberark.conjur.api.Credentials;
import com.cyberark.conjur.api.Endpoints;
import com.cyberark.conjur.api.clients.AuthnClient;
import com.cyberark.conjur.util.rs.TokenAuthFilter;

import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.deployit.util.PasswordEncrypter;
import com.xebialabs.xlrelease.domain.ExternalVariableServer;
import com.xebialabs.xlrelease.domain.variables.ExternalVariableValue;
import com.xebialabs.xlrelease.domain.variables.PasswordStringVariable;
import com.xebialabs.xlrelease.externalproviders.ExternalVariableResolutionFailedException;

@Metadata(label = "Conjur Server")
public class ConjurServer extends ExternalVariableServer {
    private static final Logger logger = LoggerFactory.getLogger(ConjurServer.class);
    @Property(description = "Script Location", required = false, hidden = true, defaultValue = "conjur/CheckConjurConnection.py")
    private String scriptLocation;

    @Property(label = "Account", required = true, category = "Authentication")
    private String account;

    @Property(label = "Username", required = true, category = "Authentication")
    private String username;

    public String getScriptLocation() {
        return scriptLocation;
    }

    public void setScriptLocation(String scriptLocation) {
        this.scriptLocation = scriptLocation;
    }

    public String getAccount() {
        return account;
    }

    public void setAccount(final String account) {
        this.account = account;
    }

    public String getUsername() {
        return username;
    }

    public void setUsername(final String username) {
        this.username = username;
    }
    //used by script under resources
    public boolean checkConnection() {
        Client client = null;
        try {
            ClientBuilder builder = createClientBuilder();
            if (builder != null) {
                client = builder.build();
                return true;
            }
            return false;
        } catch (Exception e) {
            logger.error("An error occurred when checking connection for Conjur:", e);
            return false;
        } finally {
            if (client != null) {
                client.close();
            }
        }
    }

    public ClientBuilder createClientBuilder() {
        String serverAddress = this.getUrl();
        Endpoints endpoints = createEndPoints();

        return createConnection(serverAddress, endpoints);
    }

    public Endpoints createEndPoints() {
        String serverAddress = this.getUrl();
        URI authUrl = URI.create(serverAddress + "/authn/" + account);
        URI secretsUrl = URI.create(serverAddress + "/secrets/" + account + "/variable");
        return new Endpoints(authUrl, secretsUrl);
    }

    @Override
    public Map<String, String> lookup(final List<PasswordStringVariable> variables) {

        final HashMap<String, String> lookupMap = new HashMap<>();

        ClientBuilder builder = createClientBuilder();

        Client client = null;
        String path = "";
        try {
            client = builder.build();
            WebTarget webTarget = client.target(createEndPoints().getSecretsUri());
            for (PasswordStringVariable var : variables) {
                ExternalVariableValue externalVariableValue = var.getExternalVariableValue();

                path = externalVariableValue.getPath();
                String mySecret = retrieveSecret(webTarget, path);

                lookupMap.put(var.getKey(), mySecret);
            }
        } catch (Exception e) {
            throw new ExternalVariableResolutionFailedException("Unable to communicate with Conjur server '%s'. " +
                    "Please make sure the path='%s' is correct", this.getUrl(), path);
        } finally {
            if (client != null) {
                client.close();
            }
        }

        return lookupMap;
    }

    private ClientBuilder createConnection(final String serverAddress, final Endpoints endpoints) {
        String apiToken = PasswordEncrypter.getInstance().ensureDecrypted(getToken());

        try {
            //using a lower level API here because the Conjur library will leak connection, doesn't expose the close method.
            Credentials credentials = new Credentials(username, apiToken, endpoints.getAuthnUri().toString());
            return ClientBuilder.newBuilder().register(new TokenAuthFilter(new AuthnClient(credentials, endpoints)));
        } catch (Exception e) {
            throw new ExternalVariableResolutionFailedException("Unable to communicate with Conjur server '%s'. Please make sure " +
                    "account='%s', username='%s' and token=[not shown] are all correct", serverAddress, account, username);
        }
    }

    private String retrieveSecret(final WebTarget webTarget, final String path) {
        Response response = webTarget.path(path).request().get(Response.class);

        int status = response.getStatus();
        if (status < 200 || status >= 400) {
            String errorMessage = String.format("Error code: %d, Error message: %s", status, response.readEntity(String.class));
            throw new ExternalVariableResolutionFailedException(errorMessage, status);
        }

        String value = response.readEntity(String.class);
        if (value != null) {
            return value;
        }

        throw new ExternalVariableResolutionFailedException("Retrieve secret not valid for policy -> %s ", path);
    }
}
