package com.xebialabs.xlrelease.externalproviders.vault;

import com.xebialabs.deployit.engine.spi.exception.DeployitException;
import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.deployit.util.PasswordEncrypter;
import com.xebialabs.xlrelease.domain.ExternalVariableServer;
import com.xebialabs.xlrelease.domain.variables.ExternalVariableValue;
import com.xebialabs.xlrelease.domain.variables.PasswordStringVariable;
import com.xebialabs.xlrelease.externalproviders.ExternalVariableResolutionFailedException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.StringUtils;
import org.springframework.vault.authentication.SimpleSessionManager;
import org.springframework.vault.authentication.TokenAuthentication;
import org.springframework.vault.client.RestTemplateBuilder;
import org.springframework.vault.client.VaultEndpoint;
import org.springframework.vault.client.VaultHttpHeaders;
import org.springframework.vault.core.VaultTemplate;
import org.springframework.vault.support.VaultMount;
import org.springframework.vault.support.VaultResponse;

import java.net.URI;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

@Metadata(label = "Vault Server")
public class VaultServer extends ExternalVariableServer {
    private static final String DATA = "data";
    private static final String METADATA = "metadata";
    private static final String VERSION = "version";
    private static final String TYPE_KV = "kv";
    private static final Logger logger = LoggerFactory.getLogger(VaultServer.class);

    @Property(description = "Script Location", required = false, hidden = true, defaultValue = "vault/CheckVaultConnection.py")
    private String scriptLocation;

    @Property(description = "See https://www.vaultproject.io/docs/enterprise/namespaces", required = false, category = "Enterprise")
    private String namespace;

    public String getScriptLocation() {
        return scriptLocation;
    }

    public void setScriptLocation(String scriptLocation) {
        this.scriptLocation = scriptLocation;
    }

    public String getNamespace() {
        return namespace;
    }

    public void setNamespace(String namespace) {
        this.namespace = namespace;
    }

    public VaultTemplate createConnection() {
        final String url = this.getUrl();
        final VaultEndpoint vaultEndpoint = VaultEndpoint.from(URI.create(url));
        final TokenAuthentication clientAuthentication = new TokenAuthentication(PasswordEncrypter.getInstance().ensureDecrypted(this.getToken()));

        VaultTemplate vaultTemplate;

        if (namespace != null && !namespace.isEmpty()) {
            vaultTemplate = new VaultTemplate(
                    RestTemplateBuilder.builder().endpoint(vaultEndpoint).defaultHeader(VaultHttpHeaders.VAULT_NAMESPACE, namespace),
                    new SimpleSessionManager(clientAuthentication));
        } else {
            vaultTemplate = new VaultTemplate(vaultEndpoint, clientAuthentication);
        }
        return vaultTemplate;
    }

    //used by script under resources
    public boolean checkConnection() {
        VaultTemplate vaultTemplate = null;
        try {
            vaultTemplate = createConnection();
            vaultTemplate.read("/");
            return true;
        } catch (Exception e) {
            logger.error("An error occurred when checking connection for Vault: ", e);
            return false;
        } finally {
            try {
                if (vaultTemplate != null) {
                    vaultTemplate.destroy();
                }
            } catch (Exception e) {
                logger.error("An error occurred when closing connection for Vault: ", e);

            }
        }
    }

    private Map.Entry<String, VaultMount> getVaultMount(String path, Map<String, VaultMount> allMounts) {
        // Mounts may contain path separators which make it impossible to parse the path to determine
        // what is the engine vs path.  Instead we need to loop over all mounts looking for a match
        // which is safe because vault won't let two engines exist like "myengine/foo" and "myengine"
        for (Map.Entry<String, VaultMount> entry : allMounts.entrySet()) {
            if (path.startsWith(entry.getKey())) {
                return entry;
            }
        }
        throw new ExternalVariableResolutionFailedException("No mount found for path '%s' in Vault server %s.", path, getUrl());
    }

    private String getVersion(VaultMount details) {
        String version = null;
        if (details != null) {
            String type = details.getType();
            if (type != null && type.equals(TYPE_KV)) {
                Map<String, String> options = details.getOptions();
                if (options != null) {
                    return options.get(VERSION);
                }
            }
        }
        return version;
    }

    private String getPath(String path, Map<String, VaultMount> allMounts) {
        if (path.startsWith("/")) {
            path = StringUtils.trimLeadingCharacter(path, '/');
        }

        Map.Entry<String, VaultMount> entry = getVaultMount(path, allMounts);
        String mount = entry.getKey();
        String version = getVersion(entry.getValue());

        if (version != null && version.equals("2")) {
            // version 2 has an element "data" in the path to a key that clients may
            // not know they need to add so we will handle that for them if
            // missing.  If the path also happens to contain "data" they will
            // need to actually define the correct full api.
            String subPath = path.substring(mount.length());
            if (!subPath.startsWith(DATA+"/")) {
                path = mount + DATA + "/" + subPath;
            }
        }

        return path;
    }

    @Override
    public Map<String, String> lookup(final List<PasswordStringVariable> variables) {

        final HashMap<String, String> lookupMap = new HashMap<>();

        VaultTemplate vaultTemplate = createConnection();
        Map<String, VaultMount> mounts = vaultTemplate.opsForSys().getMounts();
        String externalKey = "";
        String path = "";
        try {
            for (PasswordStringVariable var : variables) {
                ExternalVariableValue externalVariableValue = var.getExternalVariableValue();
                externalKey = externalVariableValue.getExternalKey();
                path = getPath(externalVariableValue.getPath(), mounts);
                VaultResponse response = assertNotNull(vaultTemplate.read(path), "Path not found.");

                Map<String, Object> data = assertNotNull(getResponseData(response),
                        "Cannot find data is response from path.");

                Object retrievedValue = assertNotNull(data.get(externalKey),"Key not found.");

                lookupMap.put(var.getKey(), retrievedValue.toString());
            }
        } catch (ExternalVariableResolutionFailedException e) {
            // already formatted so don't change it
            throw e;
        } catch (Throwable t) {
            // Just in case something throws unexpectedly catch and include the exception message
            throw new ExternalVariableResolutionFailedException("Unable to lookup path '%s' and key '%s' in Vault server %s; reason: %s",
                    path, externalKey, getUrl(), t.getMessage());
        } finally {
            try {
                vaultTemplate.destroy();
            } catch (Exception e) {
                logger.error(String.format("Unable to close the vault endpoint to [%s]", getUrl()), e);
            }
        }
        return lookupMap;
    }

    private Map<String, Object> getResponseData(VaultResponse response) {
        final Map<String, Object> data = response.getData();

        if (data == null) {
            return null;
        }

        //V1 vault server response
        if (!data.containsKey(METADATA)) {
            return data;
        }

        //v2 vault server response
        //noinspection unchecked
        return (Map<String, Object>) data.get(DATA);
    }

    private <T> T assertNotNull(T object, String message, Object... args) {
        if (object != null) {
            return object;
        }

        throw new DeployitException(message, args);
    }
}
