package com.xebialabs.xlrelease.reports.audit

import com.google.common.base.Strings
import com.xebialabs.xlrelease.domain._
import com.xebialabs.xlrelease.reports.domain.MaybeData._
import com.xebialabs.xlrelease.reports.domain._
import com.xebialabs.xlrelease.reports.domain.exceptions.ItemNotFoundException
import com.xebialabs.xlrelease.variable.VariableHelper
import com.xebialabs.xlrelease.variable.VariableHelper.filterOutBlankValues

import java.util
import scala.jdk.CollectionConverters._
import scala.util.{Failure, Success}

object CommonFormat {

  val MAXIMUM_CELL_LENGTH = 32767
  val ACTIVITY_LOG_RECORD_NOT_FOUND_MESSAGE = "Activity log record not found"
  val DATE_FORMAT = "mmmm d, yyyy hh:mm AM/PM"

  val DATE_WIDTH = 30

  type AssignedUser = Option[String]
  type AssignedTeam = Option[String]
  type AssignedUserAndTeam = (AssignedUser, AssignedTeam)
  type CompletedBy = Option[String]

  type PropertyName = String
  type PreResolvedProperties = Map[PropertyName, Maybe[String]]

  case class TaskAgent(assignedUserAndTeam: AssignedUserAndTeam,
                       completedBy: CompletedBy,
                       preResolvedProperties: PreResolvedProperties)

  object TaskAgent {
    def empty: TaskAgent =
      new TaskAgent(Option.empty -> Option.empty, Option.empty, Map.empty)

    def apply(assignedUserAndTeam: AssignedUserAndTeam, completedBy: CompletedBy): TaskAgent =
      new TaskAgent(assignedUserAndTeam, completedBy, Map.empty)
  }

  // make it easier to build a TaskAgent from java
  def taskAgent(user: String, team: String, completedBy: String): TaskAgent =
    TaskAgent(Option(user) -> Option(team), Option(completedBy))
  def taskAgent(user: String, team: String, completedBy: String, preResolvedProperties: java.util.Map[String, Maybe[String]]): TaskAgent =
    taskAgent(user, team, completedBy).copy(preResolvedProperties = preResolvedProperties.asScala.toMap)

  def getPlanItemTitle(release: Release, item: PlanItem): String = {
    val title: String = item match {
      case task: Task =>
        def getTaskGroupTitle(container: TaskContainer, currentTitle: String): String = {
          container match {
            case taskGroup: TaskGroup =>
              getTaskGroupTitle(taskGroup.getContainer, s"${taskGroup.getTitle} / $currentTitle")
            case _ => currentTitle
          }
        }

        getTaskGroupTitle(task.getContainer, task.getTitle)
      case _ => item.getTitle
    }

    val variables = filterOutBlankValues(release.getAllVariableValuesAsStringsWithInterpolationInfo)
    VariableHelper.replaceAllWithInterpolation(title, variables, new util.HashSet(), freezeEvenIfUnresolved = false)
  }

  def getTaskType(task: Task): String = {
    val taskType = task match {
      case customScriptTask: CustomScriptTask => customScriptTask.getPythonScript.getType
      case _ => task.getType
    }

    val typeLabel = taskType.getDescriptor.getLabel
    if (Strings.isNullOrEmpty(typeLabel)) {
      task.getType.getName
    } else {
      typeLabel
    }
  }

  def getComments(task: Task): String = {
    task.getComments.asScala.map(entry => s"${entry.getAuthor}: ${entry.getText}").mkString("\n")
  }

  def getComment(comment: Comment): String = {
    val author = if (comment.getAuthor != null) comment.getAuthor else "SYSTEM"
    val commentText =
      s"""$author - ${comment.getCreationDate}
         |${comment.getText}""".stripMargin
    commentText.substring(0, Math.min(commentText.length, MAXIMUM_CELL_LENGTH))
  }

  def getIsDelayed(task: Task): String = {
    if (task.getDelayedCount > 0) {
      "Yes"
    } else {
      "No"
    }
  }

  def getTaskOwner(release: Release, task: Task): AssignedUser = {
    Option(VariableHelper.replaceAllWithInterpolation(task.getOwner, release.getAllVariableValuesAsStringsWithInterpolationInfo, new util.HashSet[String], freezeEvenIfUnresolved = false))
  }

  def getTaskTeam(task: Task): AssignedTeam = {
    Option(task.getTeam)
  }

  def formatStartedByTrigger(startedByTrigger: StartedByTrigger): Maybe[String] = {
    startedByTrigger.trigger match {
      case Success(trigger) =>
        MaybeData.success(s"${trigger.getTitle} (${trigger.getType.getPrefix.capitalize}: ${trigger.getType.getName.capitalize})")
      case Failure(e) => startedByTrigger.triggerTag match {
        case Some(triggerTag) => MaybeData.partial(e, s"Trigger [$triggerTag]")
        case None => MaybeData.failure(e)
      }
    }
  }

  def formatFolderPath(path: Seq[String]): String =
    path.drop(1).mkString(" / ")

  def formatStartedByMasterRelease(startedByCreateReleaseTask: StartedByCreateReleaseTask): Maybe[String] =
    MaybeData.recoverWith(
      startedByCreateReleaseTask.masterReleaseTitle,
      _ -> s"[${startedByCreateReleaseTask.masterReleaseId}]"
    )

  def formatStartedByUser(startedBy: Option[UserThatCreatedOrStartedRelease]): Maybe[String] =
    startedBy.map(u => MaybeData(u.fullName.filter(_.nonEmpty).getOrElse(u.username)))
      .getOrElse(MaybeData.failure(ItemNotFoundException(ACTIVITY_LOG_RECORD_NOT_FOUND_MESSAGE)))

  def formatStartedBy(startedBy: Option[ReleaseStartedBy]): Maybe[String] = {
    startedBy.map {
      case startedByUser: StartedByUser => formatStartedByUser(startedByUser.startedBy)
      case startedByTrigger: StartedByTrigger => formatStartedByTrigger(startedByTrigger)
      case startedByCreateReleaseTask: StartedByCreateReleaseTask => formatStartedByMasterRelease(startedByCreateReleaseTask)
    }.getOrElse(MaybeData.success(""))
  }

  def formatCreatedByUser(createdBy: Option[UserThatCreatedOrStartedRelease]): Maybe[String] = {
    createdBy.map(u => MaybeData.apply(u.fullName.filter(_.nonEmpty).getOrElse(u.username)))
      .getOrElse(MaybeData.failure(ItemNotFoundException(ACTIVITY_LOG_RECORD_NOT_FOUND_MESSAGE, null)))
  }

  def prettifyStatusString(status: String): String = {
    status.capitalize.replace("_", " ")
  }

  def getMaybeReleaseTitle(templateInfo: Maybe[Option[ReleaseTitleAndPath]]): Maybe[String] = {
    templateInfo.mapValue(_.map(_.title).getOrElse(""))
  }

  def getMaybeReleasePath(templateInfo: Maybe[Option[ReleaseTitleAndPath]]): Maybe[String] = {
    templateInfo.flatMap(_.getValue.map(_.path).getOrElse(MaybeData.success("")))
  }
}
