package com.xebialabs.xlrelease.reports.repository

import com.xebialabs.xlrelease.db.sql.LimitOffset
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.reports.domain.PermissionLabels
import com.xebialabs.xlrelease.security.sql.db.SecuritySchema.{ROLES, ROLE_PERMISSIONS, ROLE_PRINCIPALS}
import com.xebialabs.xlrelease.service.UserProfileService
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.JdbcTemplate
import org.springframework.stereotype.Repository

import java.sql.ResultSet
import scala.collection.mutable
import scala.jdk.CollectionConverters._

case class GlobalUserPermissionsReportData(principal: String, userFullName: String, permission: String, roles: mutable.Buffer[String])

case class FolderUserPermissionsReportData(
                                            folderName: String,
                                            principal: String,
                                            userFullName: String,
                                            permission: String,
                                            links: List[String])

@Repository
class UserPermissionsReportsRepository @Autowired()(@Qualifier("xlrRepositoryJdbcTemplate") jdbcTemplate: JdbcTemplate,
                                                    userProfileService: UserProfileService,
                                                    @Qualifier("xlrRepositorySqlDialect") implicit val dialect: Dialect)
  extends Logging with LimitOffset {

  def generateGlobalReportData(limit: Option[Long] = None): List[GlobalUserPermissionsReportData] = {
    val STMT_SELECT_GLOBAL_REPORT =
      s"""SELECT rPrincipal.${ROLE_PRINCIPALS.principalName},
         |       rPermission.${ROLE_PERMISSIONS.permissionName},
         |       role.${ROLES.name}
         |FROM ${ROLE_PRINCIPALS.TABLE} rPrincipal
         |         LEFT JOIN ${ROLES.TABLE} role ON role.${ROLES.id} = rPrincipal.${ROLE_PRINCIPALS.roleId}
         |         INNER JOIN ${ROLE_PERMISSIONS.TABLE} rPermission ON rPermission.${ROLE_PERMISSIONS.roleId} = rPrincipal.${ROLE_PRINCIPALS.roleId}
         |WHERE role.${ROLES.ciId} = -1
         |ORDER BY rPrincipal.${ROLE_PRINCIPALS.principalName}, rPermission.${ROLE_PERMISSIONS.permissionName}, role.${ROLES.name}""".stripMargin

    val query = addLimitAndOffset(STMT_SELECT_GLOBAL_REPORT, limit, None)

    var lastAdded: GlobalUserPermissionsReportData = GlobalUserPermissionsReportData(null, null, null, null)
    jdbcTemplate.query[GlobalUserPermissionsReportData](query, (rs: ResultSet, _: Int) => {
      val principalName = rs.getString(ROLE_PRINCIPALS.principalName)
      val userFullName = Option(userProfileService.findByUsername(principalName)).map(_.getFullName).getOrElse("")
      val permissionLabel = PermissionLabels.getLabel(rs.getString(ROLE_PERMISSIONS.permissionName))
      val role = rs.getString(ROLES.name)
      if (lastAdded.principal == principalName && lastAdded.permission == permissionLabel) {
        lastAdded.roles += role
        null
      } else {
        lastAdded = GlobalUserPermissionsReportData(principalName, userFullName, permissionLabel, mutable.Buffer(role))
        lastAdded
      }
    }).asScala
      .filterNot(_ == null)
      .toList
  }

}
