package com.xebialabs.xlrelease.reports.repository.itsm

import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlrelease.api.v1.forms.FacetFilters
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.domain.facet.Facet
import com.xebialabs.xlrelease.domain.udm.reporting.ItsmRecord
import com.xebialabs.xlrelease.domain.{Release, Task}
import com.xebialabs.xlrelease.reports.repository.CommonRowAndParamsMapper._
import com.xebialabs.xlrelease.reports.repository.itsm.ItsmRecordLiveRepository._
import com.xebialabs.xlrelease.reports.repository.{RepositoryExceptionUtils, SearchRecordsSupport}
import com.xebialabs.xlrelease.repository.FacetRepository.SpecializedFacetRepository
import com.xebialabs.xlrelease.repository.Ids.{getFolderlessId, getName}
import com.xebialabs.xlrelease.repository.sql.persistence.CiId._
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.{FOLDERS, RELEASES, TASKS}
import com.xebialabs.xlrelease.repository.sql.persistence.Utils._
import com.xebialabs.xlrelease.repository.sql.persistence.{CiUid, PersistenceSupport, TaskPersistence}
import com.xebialabs.xlrelease.repository.{FacetRepository, Ids}
import com.xebialabs.xlrelease.service.CiIdService
import com.xebialabs.xlrelease.utils.FolderId
import grizzled.slf4j.Logging
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}

import java.sql.ResultSet
import scala.collection.mutable

object ItsmRecordLiveRepository {

  import ITSM_TASK_REPORTING_RECORD_LIVE._

  private lazy val ALL_FIELDS =
    s"""| ${commonFieldNames(ITSM_TASK_REPORTING_RECORD_LIVE, "i")},
        | f.${FOLDERS.FOLDER_PATH},
        | f.${FOLDERS.FOLDER_ID},
        | r.${RELEASES.RELEASE_ID},
        | t.${TASKS.TASK_ID},
        | i.$RECORD,
        | i.$RECORD_URL,
        | i.$TITLE,
        | i.$STATUS,
        | i.$PRIORITY,
        | i.$CREATED_BY""".stripMargin

  private lazy val SELECT_ALL_ITSM_RECORDS: String =
    s"""|SELECT
        | $ALL_FIELDS
        |FROM ${TABLE} i
        |JOIN ${TASKS.TABLE} t ON t.${TASKS.CI_UID} = i.$TASK_UID
        |JOIN ${RELEASES.TABLE} r ON r.${RELEASES.CI_UID} = t.${TASKS.RELEASE_UID}
        |LEFT OUTER JOIN ${FOLDERS.TABLE} f ON f.${FOLDERS.CI_UID} = r.${RELEASES.FOLDER_CI_UID}""".stripMargin

  private lazy val SELECT_ITSM_RECORD_BY_ID: String =
    s"""|$SELECT_ALL_ITSM_RECORDS
        |WHERE i.$RECORD_ID = :recordId""".stripMargin

  private lazy val SELECT_ITSM_RECORDS_BY_TASK_ID: String =
    s"""|$SELECT_ALL_ITSM_RECORDS
        |WHERE t.${TASKS.TASK_ID} = :taskId""".stripMargin

  private lazy val SELECT_ITSM_RECORDS_BY_RELEASE_ID: String =
    s"""|$SELECT_ALL_ITSM_RECORDS
        |WHERE r.${RELEASES.RELEASE_ID} = :releaseId""".stripMargin


  private lazy val EXISTS_ITSM_RECORD_BY_ID: String =
    s"""SELECT COUNT($RECORD_ID) FROM ${TABLE} WHERE $RECORD_ID = :recordId""".stripMargin

  private lazy val INSERT_ITSM_RECORD: String =
    s"""|INSERT INTO ${TABLE} (
        | ${commonFieldNames(ITSM_TASK_REPORTING_RECORD_LIVE)},
        | $TASK_UID,
        | $RECORD,
        | $RECORD_URL,
        | $TITLE,
        | $STATUS,
        | $PRIORITY,
        | $CREATED_BY
        |) VALUES (
        | ${commonParameterNames()},
        | :taskUid,
        | :record,
        | :recordUrl,
        | :title,
        | :status,
        | :priority,
        | :createdBy
        |)""".stripMargin

  private lazy val DELETE_ITSM_RECORD: String =
    s"""|DELETE FROM ${TABLE}
        |WHERE $RECORD_ID = :recordId""".stripMargin

}

class ItsmRecordLiveRepository(taskPersistence: TaskPersistence,
                               val ciIdService: CiIdService,
                               val jdbcTemplate: JdbcTemplate,
                               val dialect: Dialect)
  extends SpecializedFacetRepository(Type.valueOf(classOf[ItsmRecord]))
    with FacetRepository.ForLive
    with PersistenceSupport
    with SearchRecordsSupport[ItsmRecord]
    with RepositoryExceptionUtils
    with Logging {

  def get(recordId: String, recordType: Option[Type]): Facet = {
    recordType.foreach(requiresInstanceOfSupportedType)
    sqlQuery(SELECT_ITSM_RECORD_BY_ID, params("recordId" -> getName(recordId.normalized)), itsmRowMapper)
      .headOption.getOrElse(throw new NotFoundException(s"Record '$recordId' not found."))
  }

  override protected def doCreate(facet: Facet): Facet = {
    logger.debug("Creating ITSM record in live")
    requiresInstanceOfSupportedType(facet.getType)
    withTargetTaskUid(facet.getTargetId) { taskUid =>
      val itsmRecord = facet.asInstanceOf[ItsmRecord]
      sqlInsert(INSERT_ITSM_RECORD, asQueryParameters(itsmRecord, taskUid))
      facet.setTargetId(facet.getTargetId)
      facet
    }.getOrElse {
      throw targetTaskNotFoundException(facet.getId, facet.getTargetId)
    }
  }

  def delete(recordId: String, recordType: Option[Type]): Unit =
    sqlUpdate(DELETE_ITSM_RECORD, params("recordId" -> getName(recordId.normalized)), _ => ())

  def update(record: Facet): Facet = {
    throw new UnsupportedOperationException("You cannot update records!")
  }

  def exists(recordId: String, recordType: Option[Type]): Boolean = {
    recordType.foreach(requiresInstanceOfSupportedType)
    sqlQuery(EXISTS_ITSM_RECORD_BY_ID, params("recordId" -> getName(recordId.normalized)), _.getInt(1) > 0).headOption.exists(identity)
  }

  def findAllFacetsByRelease(release: Release): Seq[Facet] = {
    findAllFacetsByReleaseId(release.getId).toSeq
  }

  def findAllFacetsByTask(task: Task): Seq[Facet] =
    findAllFacetsByTargetId(task.getId).toSeq

  def search(facetsFilters: FacetFilters): Seq[Facet] = searchRecords(facetsFilters, SELECT_ALL_ITSM_RECORDS, itsmRowMapper)

  override protected def findAllFacetsByTargetId(targetId: CiId): mutable.Seq[ItsmRecord] =
    sqlQuery(SELECT_ITSM_RECORDS_BY_TASK_ID, params("taskId" -> getFolderlessId(targetId)), itsmRowMapper)

  override protected def findAllFacetsByReleaseId(releaseId: CiId): mutable.Seq[ItsmRecord] =
    sqlQuery(SELECT_ITSM_RECORDS_BY_RELEASE_ID, params("releaseId" -> getName(releaseId.normalized)), itsmRowMapper)

  private def withTargetTaskUid[A](targetId: CiId)(f: CiUid => A): Option[A] =
    taskPersistence.getTaskUidById(targetId).map(f)

  private val itsmRowMapper: RowMapper[ItsmRecord] = (rs: ResultSet, _: Int) => {
    val itsmRecord: ItsmRecord = Type.valueOf(rs.getString(ITSM_TASK_REPORTING_RECORD_LIVE.RECORD_TYPE))
      .getDescriptor.newInstance[ItsmRecord](rs.getString(ITSM_TASK_REPORTING_RECORD_LIVE.RECORD_ID))
    val folderPath = Option(rs.getString(FOLDERS.FOLDER_PATH)).map(FolderId.apply).getOrElse(FolderId.Root)
    val releaseId = rs.getString(RELEASES.RELEASE_ID)
    val taskId = Ids.getReleaselessChildId(rs.getString(TASKS.TASK_ID))
    val targetId = folderPath / rs.getString(FOLDERS.FOLDER_ID) / releaseId / taskId
    itsmEntryRowMapper(rs, itsmRecord, targetId.absolute, ITSM_TASK_REPORTING_RECORD_LIVE)
  }

}
