package com.xebialabs.xlrelease.reports.audit

import com.xebialabs.xlrelease.api.v1.forms.TimeFrame
import com.xebialabs.xlrelease.reports.excel.MultiReleaseAuditReport.FilterData
import com.xebialabs.xlrelease.reports.filters.ReportFilter
import com.xebialabs.xlrelease.repository.FolderRepository
import com.xebialabs.xlrelease.udm.reporting.filters.CompositeFilter
import com.xebialabs.xlrelease.udm.reporting.filters.impl.TagsFilter.Operator
import com.xebialabs.xlrelease.udm.reporting.filters.impl._
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service

import java.time.{Instant, LocalDate, ZoneId}
import java.util.Date
import scala.jdk.CollectionConverters._

@Service
class ReportFilterMapper @Autowired()(folderRepository: FolderRepository) {


  def map(filters: Seq[ReportFilter]): Seq[FilterData] = {
    filters.sortBy(dateFilterFirst).flatMap(mapReportFilter)
  }

  private def mapReportFilter: ReportFilter => Seq[FilterData] =
    mapDateFilter orElse mapSimpleFilter orElse mapCompositeFilter orElse { case _ => Seq.empty[FilterData] }

  private def mapDateFilter: PartialFunction[ReportFilter, Seq[FilterData]] = {
    case f: DateFilter =>
      f.getTimeFrame match {
        case TimeFrame.LAST_YEAR =>
          Seq(
            FilterData("From", Seq(Date.from(LocalDate.now().minusYears(1).atStartOfDay(ZoneId.systemDefault()).toInstant))),
            FilterData("To", Seq(Date.from(Instant.now())))
          )
        case TimeFrame.LAST_SIX_MONTHS =>
          Seq(
            FilterData("From", Seq(Date.from(LocalDate.now().minusMonths(6).atStartOfDay(ZoneId.systemDefault()).toInstant))),
            FilterData("To", Seq(Date.from(Instant.now())))
          )
        case TimeFrame.LAST_THREE_MONTHS =>
          Seq(
            FilterData("From", Seq(Date.from(LocalDate.now().minusMonths(3).atStartOfDay(ZoneId.systemDefault()).toInstant))),
            FilterData("To", Seq(Date.from(Instant.now())))
          )
        case TimeFrame.LAST_MONTH =>
          Seq(
            FilterData("From", Seq(Date.from(LocalDate.now().minusMonths(1).atStartOfDay(ZoneId.systemDefault()).toInstant))),
            FilterData("To", Seq(Date.from(Instant.now())))
          )
        case TimeFrame.RANGE =>
          Seq(
            FilterData("From", Seq(f.getFrom)),
            FilterData("To", Seq(f.getTo))
          )
      }
  }

  private def mapSimpleFilter: PartialFunction[ReportFilter, Seq[FilterData]] = {
    case f: FolderFilter =>
      Seq(FilterData("Parent folder matches", Seq(getFolderPath(f.getFolderId))))
    case f: ReleaseFilter =>
      Seq(FilterData("Release title contains", Seq(f.getReleaseTitle)))
    case f: TagsFilter =>
      val releaseTagsTitle = if (f.getOperator.equals(Operator.CONTAINS_ALL)) {
        "Release tags contain"
      } else {
        "Release tags (any)"
      }
      Seq(FilterData(releaseTagsTitle, f.getTags.asScala.toSeq))
    case f: ChangeNumbersFilter =>
      Seq(FilterData("Change number matches", f.getChangeNumbers.asScala.toSeq))
    case f: ApplicationNamesFilter =>
      Seq(FilterData("Application matches", f.getApplicationNames.asScala.toSeq))
    case f: EnvironmentNamesFilter =>
      Seq(FilterData("Environment matches", f.getEnvironmentNames.asScala.toSeq))
  }

  private def mapCompositeFilter: PartialFunction[ReportFilter, Seq[FilterData]] = {
    case f: CompositeFilter =>
      val mappedFilters = map(f.getFilters.asScala.toSeq)
      mappedFilters.headOption match {
        case Some(value) => Seq(FilterData(value.label, mappedFilters.flatMap(_.values)))
        case None => Seq.empty
      }
  }

  private def dateFilterFirst: ReportFilter => Integer = {
    case _: DateFilter => 0
    case _ => 1
  }

  private def getFolderPath(folderId: String): String = {
    CommonFormat.formatFolderPath(folderRepository.getPath(folderId))
  }


}
