package com.xebialabs.xlrelease.reports.db

import com.xebialabs.deployit.security.{PermissionEnforcer, RoleService}
import com.xebialabs.xlrelease.api.v1.forms.TimeFrame
import com.xebialabs.xlrelease.db.ArchivedReleases._
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.archiving.SelectArchivedReleasesBuilder
import com.xebialabs.xlrelease.domain.status.ReleaseStatus.COMPLETED
import com.xebialabs.xlrelease.reports.dto.{AverageAndLongestReleaseDuration, CompletedReleases, ReleaseAutomationData, ReleaseDuration}
import com.xebialabs.xlrelease.reports.filters.ReportFilter
import com.xebialabs.xlrelease.reports.service.{ReportParams, ReportService}
import com.xebialabs.xlrelease.serialization.json.utils.JsonWithCachedProvider
import com.xebialabs.xlrelease.service.ArchivingService.getMonthYear
import com.xebialabs.xlrelease.views.Point
import grizzled.slf4j.Logging
import org.joda.time.DateTime
import org.joda.time.Months.monthsBetween
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.JdbcTemplate
import org.springframework.stereotype.Repository
import org.springframework.util.StreamUtils.copyToString

import java.io.StringReader
import java.lang.{Iterable => JIterable}
import java.nio.charset.StandardCharsets.UTF_8
import java.sql.ResultSet
import java.text.DateFormatSymbols
import java.util.{Date, Locale, List => JList}
import javax.json.{JsonObject, JsonValue}
import scala.jdk.CollectionConverters._
import scala.util.Using

object ArchivedReleasesReports {
  private lazy val months = new DateFormatSymbols().getMonths

  private val archivedReleasesString: String = Seq(
    REPORT_RELEASES_RELEASEJSON_COLUMN,
    REPORT_RELEASES_ID_COLUMN,
    REPORT_RELEASES_END_DATE_COLUMN,
    REPORT_RELEASES_TITLE_COLUMN,
    REPORT_RELEASES_DURATION_COLUMN
  ).map(REPORT_RELEASES_TABLE_ALIAS + "." + _).mkString(", ")

  private val getReleaseAutomationString: String = Seq(
    REPORT_RELEASES_ID_COLUMN, REPORT_RELEASES_TITLE_COLUMN, REPORT_RELEASES_END_DATE_COLUMN,
    REPORT_RELEASES_MANUAL_TASKS_COUNT_COLUMN, REPORT_RELEASES_AUTOMATED_TASKS_COUNT_COLUMN,
    REPORT_RELEASES_MANUAL_TASKS_DURATION_COLUMN, REPORT_RELEASES_AUTOMATED_TASKS_DURATION_COLUMN
  ).map(REPORT_RELEASES_TABLE_ALIAS + "." + _).mkString(", ")

  def getJsonFromBlob(rs: ResultSet, columnName: String): JsonObject = {
    val binaryStream = rs.getBinaryStream(columnName)
    val content = try {
      copyToString(binaryStream, UTF_8)
    } finally {
      binaryStream.close()
    }
    val jsonObject = Using(JsonWithCachedProvider.createReader(new StringReader(content))) { reader =>
      reader.readObject()
    }
    jsonObject.getOrElse(JsonValue.EMPTY_JSON_OBJECT)
  }
}

@Repository
class ArchivedReleasesReports @Autowired()(@Qualifier("reportingJdbcTemplate") jdbcTemplate: JdbcTemplate,
                                           @Qualifier("reportingSqlDialect") implicit val reportingSqlDialect: Dialect,
                                           implicit val permissionEnforcer: PermissionEnforcer,
                                           implicit val roleService: RoleService,
                                           archivedTasksReports: ArchivedTasksReports) extends Logging {

  import ArchivedReleasesReports._

  def getReleaseDuration(reportParams: ReportParams): JList[Point] = {
    val (startDate, endDate) = getStartAndEndDates(reportParams.timeFrame, reportParams.from, reportParams.to)
    val (sql, params) = buildQueryGroupedByMonth(reportParams.tags, startDate, endDate, reportParams.userSpecific, reportParams.filters,
      s"AVG($REPORT_RELEASES_DURATION_COLUMN) AS average", REPORT_RELEASES_MONTH_YEAR_COLUMN
    )

    val presentMonths: Map[String, Float] = jdbcTemplate.query(sql, params.toArray, (rs: ResultSet, _: Int) =>
      (rs.getString(REPORT_RELEASES_MONTH_YEAR_COLUMN), rs.getFloat("average"))
    ).asScala.toMap

    if (presentMonths.isEmpty) {
      List.empty[Point]
    } else {
      completeSeries(startDate, endDate, (monthIndex: Int, _: DateTime, monthYear: String) => {
        val monthName = months(monthYear.split('.').head.toInt - 1)
        PointOption(monthName, monthIndex + 1, presentMonths.get(monthYear))
      })
    }
  }.asJava

  def getLongestReleases(reportParams: ReportParams): JList[ReleaseDuration] = {
    val sqlQuery = new SelectArchivedReleasesBuilder(archivedReleasesString)
      .withDates(reportParams.timeFrame, reportParams.from, reportParams.to)
      .withTags(reportParams.tags)
      .withSecurity(reportParams.userSpecific)
      .withNotNull(REPORT_RELEASES_DURATION_COLUMN)
      .withOneOfStatuses(COMPLETED.value())
      .withFilters(reportParams.filters)
      .orderBy(s"$REPORT_RELEASES_DURATION_COLUMN DESC")
      .limit(ReportService.TOP_REPORT_SIZE)

    val (sql, params) = sqlQuery.build()
    jdbcTemplate.query(sql, params.toArray, (rs: ResultSet, _: Int) => {
      val releaseJson = getJsonFromBlob(rs, REPORT_RELEASES_RELEASEJSON_COLUMN)
      val releaseId = if (releaseJson.containsKey("id")) releaseJson.getString("id") else rs.getString(REPORT_RELEASES_ID_COLUMN)
      val timestamp = rs.getTimestamp(REPORT_RELEASES_END_DATE_COLUMN)
      val releaseDuration = new ReleaseDuration(releaseId, rs.getString(REPORT_RELEASES_TITLE_COLUMN),
        new DateTime(timestamp.getTime), rs.getLong(REPORT_RELEASES_DURATION_COLUMN))
      releaseDuration
    })
  }

  def getReleaseAutomation(reportParams: ReportParams): JList[ReleaseAutomationData] = {
    val sqlQuery = new SelectArchivedReleasesBuilder(getReleaseAutomationString)
      .withDates(reportParams.timeFrame, reportParams.from, reportParams.to)
      .withTags(reportParams.tags)
      .withSecurity(reportParams.userSpecific)
      .withNotNull(REPORT_RELEASES_END_DATE_COLUMN)
      .withOneOfStatuses(COMPLETED.value())
      .withFilters(reportParams.filters)
      .orderBy(s"$REPORT_RELEASES_END_DATE_COLUMN DESC")
    if (reportParams.limit != null) {
      sqlQuery.limit(reportParams.limit.longValue())
    }

    val (sql, params) = sqlQuery.build()
    jdbcTemplate.query(sql, params.toArray, (rs: ResultSet, _: Int) => {
      val manualTasksCount = rs.getInt(REPORT_RELEASES_MANUAL_TASKS_COUNT_COLUMN)
      val automatedTasksCount = rs.getInt(REPORT_RELEASES_AUTOMATED_TASKS_COUNT_COLUMN)
      val manualTasksDuration = rs.getLong(REPORT_RELEASES_MANUAL_TASKS_DURATION_COLUMN)
      val automatedTasksDuration = rs.getLong(REPORT_RELEASES_AUTOMATED_TASKS_DURATION_COLUMN)

      new ReleaseAutomationData(
        rs.getString(REPORT_RELEASES_ID_COLUMN),
        rs.getString(REPORT_RELEASES_TITLE_COLUMN),
        new DateTime(rs.getTimestamp(REPORT_RELEASES_END_DATE_COLUMN).getTime),
        manualTasksCount + automatedTasksCount,
        automatedTasksCount,
        manualTasksDuration + automatedTasksDuration,
        automatedTasksDuration
      )
    })
  }

  def getNumberOfReleaseByMonth(reportParams: ReportParams): JList[Point] = {
    val (startDate, endDate) = getStartAndEndDates(reportParams.timeFrame, reportParams.from, reportParams.to)
    val (sql, params) = buildQueryGroupedByMonth(reportParams.tags, startDate, endDate, reportParams.userSpecific,
      reportParams.filters, s"COUNT($REPORT_RELEASES_TABLE_ALIAS.$REPORT_RELEASES_ID_COLUMN) AS releaseCount", REPORT_RELEASES_MONTH_YEAR_COLUMN)

    val presentMonths: Map[String, Float] = jdbcTemplate.query(sql, params.toArray, (rs: ResultSet, _: Int) =>
      (rs.getString(REPORT_RELEASES_MONTH_YEAR_COLUMN), rs.getFloat("releaseCount"))
    ).asScala.toMap

    if (presentMonths.isEmpty) {
      List.empty[Point]
    } else {
      completeSeries(startDate, endDate, (_: Int, currentDate: DateTime, monthYear: String) => {
        val middleOfMonth = new DateTime(currentDate.getYear, currentDate.getMonthOfYear, 15, 0, 0).getMillis
        val pointTitle = currentDate.toString("MMMM", Locale.ENGLISH)
        PointOption(pointTitle, middleOfMonth, presentMonths.get(monthYear))
      })
    }
  }.asJava

  def getCompletedReleases(reportParams: ReportParams): CompletedReleases = {
    val sqlQuery: SelectArchivedReleasesBuilder = new SelectArchivedReleasesBuilder(
      s"""
         |COUNT($REPORT_RELEASES_TABLE_ALIAS.$REPORT_RELEASES_ID_COLUMN) AS completedReleases,
         |COUNT(DISTINCT $REPORT_RELEASES_TABLE_ALIAS.$REPORT_RELEASES_ORIGIN_TEMPLATE_ID) AS createdFromTemplate""".stripMargin)
      .withDates(reportParams.timeFrame, reportParams.from, reportParams.to)
      .withTags(reportParams.tags)
      .withSecurity(reportParams.userSpecific)
      .withOneOfStatuses(COMPLETED.value())
      .withFilters(reportParams.filters)

    val (sql, params) = sqlQuery.build()
    jdbcTemplate.queryForObject(sql, params.toArray, (rs: ResultSet, _: Int) =>
      new CompletedReleases(
        rs.getLong("completedReleases"),
        rs.getLong("createdFromTemplate")
      ))
  }

  def getAverageAndLongestReleaseDuration(reportParams: ReportParams): AverageAndLongestReleaseDuration = {
    val sqlQuery: SelectArchivedReleasesBuilder = new SelectArchivedReleasesBuilder(
      s"""
         |AVG($REPORT_RELEASES_TABLE_ALIAS.$REPORT_RELEASES_DURATION_COLUMN) AS averageReleaseDuration,
         |MAX($REPORT_RELEASES_TABLE_ALIAS.$REPORT_RELEASES_DURATION_COLUMN) AS longestDuration""".stripMargin)
      .withDates(reportParams.timeFrame, reportParams.from, reportParams.to)
      .withTags(reportParams.tags)
      .withSecurity(reportParams.userSpecific)
      .withFilters(reportParams.filters)

    val (sql, params) = sqlQuery.build()
    jdbcTemplate.queryForObject(sql, params.toArray, (rs: ResultSet, _: Int) =>
      new AverageAndLongestReleaseDuration(
        rs.getLong("averageReleaseDuration"),
        rs.getLong("longestDuration")
      ))
  }

  private def buildQueryGroupedByMonth(tags: JIterable[String], startDate: Long, endDate: Long,
                                       userSpecific: Boolean, filters: JList[ReportFilter], selectedColumns: String*) = {
    new SelectArchivedReleasesBuilder(selectedColumns: _*)
      .withStartDate(new Date(startDate))
      .withEndDate(new Date(endDate))
      .withTags(tags)
      .withSecurity(userSpecific)
      .withOneOfStatuses(COMPLETED.value())
      .withNotNull(REPORT_RELEASES_MONTH_YEAR_COLUMN)
      .withFilters(filters)
      .groupBy(REPORT_RELEASES_MONTH_YEAR_COLUMN)
      .build()
  }

  private def getStartAndEndDates(timeFrame: TimeFrame, from: Date, to: Date): (Long, Long) =
    (timeFrame.getStartDate(from), timeFrame.getEndDate(to))

  private def completeSeries(startDate: Long, endDate: Long,
                             makePoint: (Int, DateTime, String) => PointOption): List[Point] = {
    val monthsCount = monthsBetween(new DateTime(startDate), new DateTime(endDate)).getMonths
    val serie = 0 to monthsCount map { monthIndex =>
      val currentDate = new DateTime(startDate).plusMonths(monthIndex)
      val monthYear = getMonthYear(currentDate)
      makePoint(monthIndex, currentDate, monthYear)
    }

    val trimmedSerie = serie
      .dropWhile(p => p.y.isEmpty)
      .reverse.dropWhile(p => p.y.isEmpty).reverse

    trimmedSerie.map(p => new Point(p.name, p.x, p.y.getOrElse(0f))).toList
  }

  case class PointOption(name: String, x: Float, y: Option[Float])

}
