package com.xebialabs.xlrelease.reports.job.impl

import akka.actor.{Actor, ActorLogging, ActorRef}
import akka.cluster.pubsub.DistributedPubSub
import akka.cluster.pubsub.DistributedPubSubMediator.Publish
import com.xebialabs.xlrelease.reports.job.api.ReportingEngineService.ReportJobId
import org.springframework.security.core.Authentication
import org.springframework.security.core.context.SecurityContextHolder

import scala.collection.mutable

object ReportJobDelegateActor {

  sealed trait ReportJobDelegateActorCommand {
    val callerContext: Option[Authentication] = Option(SecurityContextHolder.getContext.getAuthentication)
  }

  case class Enqueue(jobId: ReportJobId, reportJobInstance: ReportJobInstance) extends ReportJobDelegateActorCommand

  case class Abort(jobId: ReportJobId) extends ReportJobDelegateActorCommand

  case class AbortRequested(jobId: ReportJobId) extends ReportJobDelegateActorCommand

  val name = "reportJobDelegateActor"
}

trait BaseReportJobDelegateActor extends Actor with ActorLogging {

  import ReportJobDelegateActor._

  private val reportJobFutures: mutable.Map[ReportJobId, ReportJobInstance] = mutable.Map.empty

  protected def enqueue(jobId: ReportJobId, reportJobInstance: ReportJobInstance): Option[ReportJobInstance] = {
    reportJobFutures.put(jobId, reportJobInstance)
  }

  private def remove(jobId: ReportJobId): Option[ReportJobInstance] = {
    reportJobFutures.remove(jobId)
  }

  protected def doAbort(jobId: ReportJobId): Unit = {
    remove(jobId) match {
      case Some(reportJobInstance) =>
        reportJobInstance.abort()
      case None =>
        log.warning(s"Could not find JobFuture for ReportJobId '${jobId}'. It probably lives on another node.")
    }
  }

  protected def withCallerContext(delegate: Receive): Receive = {
    case cmd: ReportJobDelegateActorCommand =>
      val maybeAuthentication = cmd.callerContext
      log.debug(s"Setting caller as ${maybeAuthentication.map(_.getName)} for command $cmd")
      maybeAuthentication.foreach(SecurityContextHolder.getContext.setAuthentication)
      delegate(cmd)
    case msg@_ =>
      delegate(msg)
  }
}

class ReportJobDelegateActor extends BaseReportJobDelegateActor {

  import ReportJobDelegateActor._

  override def receive: Receive = withCallerContext(singleNodeReceive)

  private def singleNodeReceive: Receive = {
    case Enqueue(jobId, reportJobInstance) =>
      enqueue(jobId, reportJobInstance)
    case Abort(jobId) => doAbort(jobId)
  }
}

class ClusteredReportJobDelegateActor extends BaseReportJobDelegateActor {

  import ReportJobDelegateActor._
  import akka.cluster.pubsub.DistributedPubSubMediator.{Subscribe, SubscribeAck}

  val topicName = "reportJobs"
  val mediator: ActorRef = DistributedPubSub(context.system).mediator
  mediator ! Subscribe(topicName, self)

  override def receive: Receive = withCallerContext(clusteredReceive)

  private def clusteredReceive: Receive = {
    case Enqueue(jobId, reportJobInstance) =>
      enqueue(jobId, reportJobInstance)
    case Abort(jobId) =>
      mediator ! Publish(topicName, AbortRequested(jobId))
    case AbortRequested(jobId) => doAbort(jobId)
    case SubscribeAck(Subscribe(topicName, None, self)) =>
      log.info(s"subscribing to topic '${topicName}'")
  }
}
