package com.xebialabs.xlrelease.reports.upgrade

import com.xebialabs.deployit.ServerConfiguration
import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.plugin.api.udm.Metadata.ConfigurationItemRoot.APPLICATIONS
import com.xebialabs.deployit.server.api.upgrade.{Upgrade, Version}
import com.xebialabs.xlrelease.db.ArchivedReleases
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.domain.udm.reporting.DeploymentRecord
import com.xebialabs.xlrelease.reports.repository.deployment.DeploymentRecordRepository
import com.xebialabs.xlrelease.reports.upgrade.XLRelease950DeploymentToRecordUpgrade.{DeploymentData, FIND_DEPLOYMENTS_BY_RELEASE_ID}
import com.xebialabs.xlrelease.repository.FacetRepository.SpecializedFacetRepository
import com.xebialabs.xlrelease.repository.IdType.DOMAIN
import com.xebialabs.xlrelease.repository.Ids
import com.xebialabs.xlrelease.repository.sql.persistence.PersistenceSupport
import com.xebialabs.xlrelease.repository.sql.persistence.Utils.params
import com.xebialabs.xlrelease.udm.reporting.repository.sql.persistence.DeploymentSchema.DEPLOYMENTS
import com.xebialabs.xlrelease.udm.reporting.repository.sql.persistence.{DeploymentHistoryPersistence, DeploymentPersistence}
import com.xebialabs.xlrelease.udm.reporting.{Deployment, DeploymentHistory}
import com.xebialabs.xlrelease.upgrade.Components.XL_RELEASE_COMPONENT
import com.xebialabs.xlrelease.upgrade.UpgradeSupport.{BatchSupport, TransactionSupport}
import grizzled.slf4j.{Logger, Logging}
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.JdbcTemplate
import org.springframework.stereotype.Component
import org.springframework.transaction.support.TransactionTemplate

import scala.util.matching.Regex

object XLRelease950DeploymentToRecordUpgrade {

  case class DeploymentData(releaseId: String, deployment: Deployment, history: Seq[DeploymentHistory])

  private lazy val logger = Logger(classOf[XLRelease950DeploymentToRecordUpgrade])

  val DEPLOYMENT_TASK_ID_REGEX: Regex = "[0-9a-fA-F]{8}\\-[0-9a-fA-F]{4}\\-[0-9a-fA-F]{4}\\-[0-9a-fA-F]{4}\\-[0-9a-fA-F]{12}".r


  def taskUrl(taskId: String)(implicit serverConfiguration: ServerConfiguration): String = {
    val fullTaskId = if (!taskId.startsWith(APPLICATIONS.getRootNodeName)) {
      APPLICATIONS.getRootNodeName + Ids.SEPARATOR + taskId
    } else {
      taskId
    }
    val releasePage = s"releases/${DOMAIN.convertToViewId(Ids.releaseIdFrom(fullTaskId))}"
    val taskDetailsId = DOMAIN.convertToViewId(fullTaskId)
    s"${serverConfiguration.getServerUrl}#/$releasePage?openTaskDetailsModal=${taskDetailsId}"
  }

  implicit class DeploymentDataOps(val _data: DeploymentData) extends AnyVal {
    def toDeploymentRecords(implicit serverConfiguration: ServerConfiguration): Seq[DeploymentRecord] = {
      for {
        deployment <- Option(_data.deployment)
        history <- Option(_data.history)
        if history.nonEmpty
      } yield {
        history.map { change =>
          logger.trace(s"change: ${change}")
          val record = Type.valueOf(classOf[DeploymentRecord]).getDescriptor.newInstance[DeploymentRecord](null)
          Option(change.getChangeDate).foreach(record.setCreationDate)
          logger.trace(s"- creationDate: '${record.getCreationDate}'")
          Option(deployment.getTaskId).foreach { taskId =>
            val fullTaskId = _data.releaseId + Ids.SEPARATOR + Ids.getReleaselessChildId(taskId)
            record.setTargetId(fullTaskId)
          }
          logger.trace(s" == ${_data.releaseId}")
          logger.trace(s"- targetId: '${record.getTargetId}'")
          Option(deployment.getFailuresCount).foreach(record.setRetryAttemptNumber)
          logger.trace(s"- retryAttemptNumber: '${record.getRetryAttemptNumber}'")
          Option(change.getDeployUrl) match {
            case None =>
              record.setDeploymentTask(record.getTargetId)
              record.setDeploymentTask_url(taskUrl(record.getTargetId))
            case Some(deployUrl) =>
              record.setDeploymentTask(DEPLOYMENT_TASK_ID_REGEX.findFirstIn(deployUrl).getOrElse("<unknown>"))
              record.setDeploymentTask_url(deployUrl)
          }
          logger.trace(s"- deploymentTask: '${record.getDeploymentTask}' @ '${record.getDeploymentTask_url}'")
          Option(deployment.getApplicationName).foreach(record.setApplicationName)
          Option(deployment.getVersion).foreach(record.setVersion)
          Option(deployment.getEnvironmentName).foreach(record.setEnvironmentName)
          logger.trace(s" - app/version/env: '${record.getApplicationName}/${record.getVersion}/${record.getEnvironmentName}'")
          Option(change.getNewStatus).foreach(record.setStatus)
          logger.trace(s" - status: '${record.getStatus}'")

          // TODO:
          // server url => if this was from XLD, possibly we *could* guesstimate it from deployUrl (minus the application-context)
          // server user => we need the task for that, let's not load it

          record
        }
      }
    }.getOrElse(Seq.empty)
  }


  private val FIND_DEPLOYMENTS_BY_RELEASE_ID: String =
    s"""|SELECT * FROM ${DEPLOYMENTS.TABLE}
        |WHERE ${DEPLOYMENTS.RELEASE_ID} = ?""".stripMargin

}

@Component
class XLRelease950DeploymentToRecordUpgrade @Autowired()(@Qualifier("reportingJdbcTemplate") val jdbcTemplate: JdbcTemplate,
                                                         @Qualifier("reportingSqlDialect") implicit val dialect: Dialect,
                                                         @Qualifier("reportTransactionTemplate") val transactionTemplate: TransactionTemplate,
                                                         implicit val serverConfiguration: ServerConfiguration,
                                                         archivedReleases: ArchivedReleases,
                                                         deploymentRecordRepository: DeploymentRecordRepository,
                                                         deploymentPersistence: DeploymentPersistence,
                                                         deploymentHistoryPersistence: DeploymentHistoryPersistence)
  extends Upgrade
    with PersistenceSupport
    with Logging
    with BatchSupport
    with TransactionSupport {

  override def upgradeVersion(): Version = Version.valueOf(XL_RELEASE_COMPONENT, "9.5.0#6")

  def doUpgrade(): Boolean = {
    doInBatch(findReleaseIdsWithDeployments(), itemsName = "Deployments") { batch =>
      batch.items.foreach { releaseId =>
        logger.trace(s"Making DeploymentRecords out of Deployments and DeploymentHistories for '$releaseId'")
        doInTransaction(migrate(releaseId))
      }
    }
    true
  }

  private def migrate(releaseId: String): Unit = {
    val repository = getRepositoryForRelease(releaseId)

    for {
      data <- findDeploymentData(releaseId)
      record <- data.toDeploymentRecords
    } {
      try {
        repository.create(record)
      } catch {
        case e: NotFoundException => logger.warn(e.getMessage)
      }
    }
  }

  private def getRepositoryForRelease(releaseId: String): SpecializedFacetRepository = {
    if (archivedReleases.exists(releaseId)) {
      deploymentRecordRepository.archiveRepository
    } else {
      deploymentRecordRepository.liveRepository
    }
  }

  def findReleaseIdsWithDeployments(): Seq[String] = {
    sqlQuery(
      s"""|SELECT DISTINCT
          | ${DEPLOYMENTS.RELEASE_ID}
          |FROM ${DEPLOYMENTS.TABLE}
          |ORDER BY
          | ${DEPLOYMENTS.RELEASE_ID} ASC""".stripMargin,
      params(),
      rs => rs.getString(DEPLOYMENTS.RELEASE_ID)
    ).toSeq
  }

  def findDeploymentData(releaseId: String): Seq[DeploymentData] = {
    val deployments: Map[String, Deployment] = deploymentPersistence.findByQuery(FIND_DEPLOYMENTS_BY_RELEASE_ID, Seq(releaseId))
      .map(d => d.getDeploymentId -> d).toMap
    val fullReleaseId = if (!releaseId.startsWith(APPLICATIONS.getRootNodeName)) {
      APPLICATIONS.getRootNodeName + Ids.SEPARATOR + releaseId
    } else {
      releaseId
    }
    val template = DeploymentData(releaseId = fullReleaseId, null, null)
    deploymentHistoryPersistence.findByIds(deployments.keySet.toSeq).groupBy(_.getDeploymentId).flatMap {
      case (deploymentId, history) => deployments.get(deploymentId).map(d => template.copy(deployment = d, history = history))
    }.toSeq
  }

}
