package com.xebialabs.xlrelease.reports.timeline

import com.xebialabs.xlrelease.api.v1.forms.ReleasesFilters
import com.xebialabs.xlrelease.domain.Phase
import com.xebialabs.xlrelease.domain.calendar.{Blackout, SpecialDay}
import com.xebialabs.xlrelease.domain.status.ReleaseStatus
import com.xebialabs.xlrelease.reports.audit.{CommonFormat, InstanceData}
import com.xebialabs.xlrelease.reports.domain.MaybeData
import com.xebialabs.xlrelease.reports.excel._
import com.xebialabs.xlrelease.reports.timeline.TimelineExport._
import org.apache.poi.ss.usermodel.Workbook
import org.apache.poi.xssf.usermodel.XSSFCellStyle

import java.util.Date
import scala.jdk.CollectionConverters._

object TimelineExport {
  val TIMELINE_EXPORT_TITLE = "Timeline export"
  val EXPORT_FILE_NAME = "timeline_report.xlsx"
  val EXPORT_CONTENT_TYPE = "application/vnd.ms-excel"
  val LOGO_WIDTH: Integer = 86
  val LOGO_HEIGHT: Integer = 9
  val NUMBER_OF_COLUMNS: Integer = 8
  val START_DATE = "Start date"
  val END_DATE = "End date"

  case class ReleaseData(title: String,
                         status: ReleaseStatus,
                         plannedStartDate: Date,
                         dueDate: Date,
                         startDate: Date,
                         endDate: Date,
                         folderName: String,
                         phases: List[Phase]
                        )

  case class Data(generatedBy: String,
                  generatedOn: Date,
                  instanceData: InstanceData,
                  releaseData: Seq[ReleaseData] = Seq.empty,
                  blackouts: Seq[Blackout],
                  labels: Seq[SpecialDay])

  def getWorkBook(data: Data, filters: ReleasesFilters): Workbook = new TimelineExport(data, filters).getWorkBook

}

class TimelineExport(data: Data, filters: ReleasesFilters) extends ExcelCommonHeader with ErrorHandler {

  override val workbook: ReportWorkbook = new ReportWorkbook
  override val styles: ExcelStyles = workbook.getStyles
  override var sheetWriter: ExcelSheetWriter = _

  def getWorkBook: Workbook = {
    generateWorksheetTitle()
    generateReleasesSheet()
    generateBlackoutsSheet()
    generateLabelsSheet()
    generateTasksSheet()

    workbook.makeWorkbookWithErrors(styles)
  }

  private def generateWorksheetTitle(): Unit = {
    sheetWriter = workbook.createReportSection("Releases")
    addCommonHeader(data.generatedBy, data.generatedOn, data.instanceData,
      NUMBER_OF_COLUMNS, LOGO_WIDTH, LOGO_HEIGHT, ReportInfo(TIMELINE_EXPORT_TITLE, infoLabel = "Export info"))
    addHeaderFilterInfo()
    addNoBorderEmptyRow(NUMBER_OF_COLUMNS)
    addNoBorderEmptyRow(NUMBER_OF_COLUMNS)
    addNoBorderEmptyRow(NUMBER_OF_COLUMNS)

    sheetWriter.newRow()
  }

  private def addReleasesSheetHeaders(): ExcelSheetWriter = {
    sheetWriter
      .addHeaderCell("Release title", 20)
      .addHeaderCell("Status", 20)
      .addHeaderCell("Planned start date", 30)
      .addHeaderCell("Due date", 28)
      .addHeaderCell(START_DATE, 28)
      .addHeaderCell(END_DATE, 28)
      .addHeaderCell("", 9)
      .addHeaderCell("", 9)
  }

  def generateReleasesSheet(): Unit = {
    addReleasesSheetHeaders()
    data.releaseData.foreach { releaseData =>
      sheetWriter
        .newRow()
        .addCell(releaseData.title)
        .addCell(CommonFormat.prettifyStatusString(releaseData.status.value()), styles.default)
      addCellForMaybe(MaybeData.allowingNull(releaseData.plannedStartDate), styles.date)
      addCellForMaybe(MaybeData.allowingNull(releaseData.dueDate), styles.date)
      addCellForMaybe(MaybeData.allowingNull(releaseData.startDate), styles.date)
      addCellForMaybe(MaybeData.allowingNull(releaseData.endDate), styles.date)
    }
  }

  def generateBlackoutsSheet(): Unit = {
    sheetWriter = workbook.createReportSection("Blackouts")
    sheetWriter
      .newRow()
      .addHeaderCell(START_DATE, 28)
      .addHeaderCell(END_DATE, 28)
      .addHeaderCell("Blackout", 28)

    data.blackouts.foreach { blackout =>
      sheetWriter
        .newRow()
        .addCell(blackout.getStartDate, styles.date)
        .addCell(blackout.getEndDate, styles.date)
        .addCell(blackout.getLabel)
    }
  }

  def generateLabelsSheet(): Unit = {
    sheetWriter = workbook.createReportSection("Labels")
    sheetWriter
      .newRow()
      .addHeaderCell("Date", 28)
      .addHeaderCell("Label", 28)

    data.labels.foreach { label =>
      sheetWriter
        .newRow()
        .addCell(label.getStartDate, styles.date)
        .addCell(label.getLabel)
    }
  }

  private def addTasksSheetHeaders(): Unit = {
    sheetWriter
      .newRow()
      .addHeaderCell("Release", 28)
      .addHeaderCell("Folder", 28)
      .addHeaderCell("Phase", 22)
      .addHeaderCell("Task", 34)
      .addHeaderCell(START_DATE, 28)
      .addHeaderCell(END_DATE, 28)
  }

  def generateTasksSheet(): Unit = {
    Option(filters.getTaskTags)
      .collect {
        case taskTags if !taskTags.isEmpty => taskTags.asScala.map(_.toLowerCase).asJava
      }
      .foreach { taskTagsToFilterOn =>
        sheetWriter = workbook.createReportSection("Tasks")
        addTasksSheetHeaders()
        for {
          release <- data.releaseData
          phase <- release.phases
          task <- phase.getTasks.asScala
          taskTagsLowercase = task.getTags.asScala.map(_.toLowerCase).asJava
          if taskTagsLowercase.containsAll(taskTagsToFilterOn)
        } {
          sheetWriter
            .newRow()
            .addCell(release.title, styles.wrapped)
            .addCell(release.folderName)
            .addCell(phase.getTitle)
            .addCell(task.getTitle, styles.wrapped)
          addCellForMaybe(MaybeData.allowingNull(task.getStartDate), styles.date)
          addCellForMaybe(MaybeData.allowingNull(task.getEndDate), styles.date)
        }
      }
  }

  private def addHeaderFilterInfo(): Unit = {
    addNoBorderEmptyRow(NUMBER_OF_COLUMNS)
    sheetWriter
      .newRow()
      .addCell("Timeline filters", styles.noGridBoldText)

    sheetWriter
      .addCell("From", styles.noGrid)
      .addCell(filters.getFrom, styles.noGridDate)
    fillCellUntilEndOfTheRow(styles.noGrid)
    fixCellPosition()
    sheetWriter
      .addCell("To", styles.noGrid)
      .addCell(filters.getTo, styles.noGridDate)
    fillCellUntilEndOfTheRow(styles.noGrid)

    if (filters.getTitle != null && !filters.getTitle.isEmpty) {
      fixCellPosition()
      sheetWriter
        .addCell("Title", styles.noGrid)
        .addCell(filters.getTitle, styles.noGrid)
      fillCellUntilEndOfTheRow(styles.noGrid)
    }

    addListFilter[String]("Release Tags", filters.getTags, tag => tag)
    addListFilter[String]("Task Tags", filters.getTaskTags, tag => tag)
    addListFilter[ReleaseStatus]("Release statuses", filters.getStatuses, status => CommonFormat.prettifyStatusString(status.value()))

    if (filters.getRiskStatusWithThresholds != null && filters.getRiskStatusWithThresholds.getRiskStatus != null) {
      fixCellPosition()
      sheetWriter
        .addCell("Risk", styles.noGrid)
        .addCell(CommonFormat.prettifyStatusString(filters.getRiskStatusWithThresholds.getRiskStatus.name), styles.noGrid)
      fillCellUntilEndOfTheRow(styles.noGrid)
    }

    fixCellPosition()
    sheetWriter
      .addCell("Only mine", styles.noGrid)
      .addCell(filters.withOnlyMine().toString, styles.noGrid)
    fillCellUntilEndOfTheRow(styles.noGrid)
    fixCellPosition()
    sheetWriter
      .addCell("Flagged", styles.noGrid)
      .addCell(filters.withOnlyFlagged().toString, styles.noGrid)
    fillCellUntilEndOfTheRow(styles.noGrid)
  }

  private def fixCellPosition() = {
    if (sheetWriter.getColumnIndex == 0) {
      sheetWriter.addEmptyCell(styles.noGrid)
    } else if (sheetWriter.getColumnIndex != 1) {
      sheetWriter.newRow().addEmptyCell(styles.noGrid)
    }
  }

  private def addListFilter[T](filterName: String, values: java.util.List[T], callback: T => String): Unit = {
    if (values != null) {
      values.asScala.zipWithIndex.foreach {
        case (value: T, 0) =>
          fixCellPosition()
          sheetWriter
            .addCell(filterName, styles.noGrid)
            .addCell(callback(value), styles.noGrid)
          fillCellUntilEndOfTheRow(styles.noGrid)
        case (value: T, _) =>
          sheetWriter
            .newRow()
            .addEmptyCell(styles.noGrid)
            .addEmptyCell(styles.noGrid)
            .addCell(callback(value), styles.noGrid)
          fillCellUntilEndOfTheRow(styles.noGrid)
      }
    }
  }

  private def fillCellUntilEndOfTheRow(style: XSSFCellStyle): Unit = {
    (sheetWriter.getColumnIndex until NUMBER_OF_COLUMNS).foreach(_ => sheetWriter.addEmptyCell(style))
  }
}
