package com.xebialabs.xlrelease.reports.audit

import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlrelease.domain.{ActivityLogEntry, ReleaseActivity}
import com.xebialabs.xlrelease.reports.audit.ActivityLogReport._
import com.xebialabs.xlrelease.reports.domain.MaybeData
import com.xebialabs.xlrelease.reports.excel._
import com.xebialabs.xlrelease.repository.Ids
import com.xebialabs.xlrelease.repository.Ids._

import scala.jdk.CollectionConverters._

object ActivityLogReport {
  val SECTION_LABEL: String = "Activity Logs"

  val DATE_COLUMN = ExcelHeaderColumn("Date", CommonFormat.DATE_WIDTH)
  val CATEGORIES_COLUMN = ExcelHeaderColumn("Categories", 18)
  val EVENT_COLUMN = ExcelHeaderColumn("Event", 30)
  val USER_COLUMN = ExcelHeaderColumn("User", 11)
  val MESSAGE_COLUMN = ExcelHeaderColumn("Message", 66)
  val ITEM_COLUMN = ExcelHeaderColumn("Item", 16)
  val TYPE_COLUMN = ExcelHeaderColumn("Type", 16)

  type UserFullName = String
  case class Data(logs: Seq[(ActivityLogEntry, UserFullName)], releaseTitle: Option[String], teams: Map[String, String],
                  phases: Map[String, String], tasks: Map[String, String])

  def addSection(data: Data, workbook: ReportWorkbook, styles: ExcelStyles): Unit = {
    new ActivityLogReport(data, workbook, styles).addContent()
  }

}

class ActivityLogReport(data: ActivityLogReport.Data,
                  val workbook: ReportWorkbook,
                  val styles: ExcelStyles) extends ErrorHandler {

  var sheetWriter: ExcelSheetWriter = workbook.createReportSection(SECTION_LABEL)

  def addContent(): Unit = {
    addHeaderRows()
    data.logs.foreach(addActivityLogRow)

  }

  private def addHeaderRows(): Unit = {
    sheetWriter.newRow()
    Seq(
      DATE_COLUMN,
      CATEGORIES_COLUMN,
      EVENT_COLUMN,
      USER_COLUMN,
      MESSAGE_COLUMN,
      ITEM_COLUMN,
      TYPE_COLUMN
    ).foreach(sheetWriter.addHeaderCell(_, styles.whiteOnGreenBig))
  }

  private def addActivityLogRow(logEntry: (ActivityLogEntry, UserFullName)): Unit = logEntry match {
    case (activityLogEntry, userFullName) =>
      val categories = ReleaseActivity.safeValueOf(activityLogEntry.getActivityType)
        .getCategories.asScala
        .mkString(", ").toLowerCase
      val targetId = activityLogEntry.getTargetId
      val itemText = if (targetId == null) {
          ""
        } else if(isTeamId(targetId)) {
          getTeamName(targetId)
        } else if(isTaskId(targetId)) {
          getTaskTitle(targetId)
        } else if(isPhaseId(targetId)) {
          getPhaseTitle(targetId)
        } else if(isReleaseId(targetId)) {
          getReleaseTitle(targetId)
        } else {
          getItemText(targetId)
        }
      sheetWriter.newRow()
      addCellForMaybe(MaybeData(activityLogEntry.getEventTime))
      addCellForMaybe(MaybeData.nonEmptyString(categories))
      addCellForMaybe(MaybeData.nonEmptyString(activityLogEntry.getActivityType))
      addCellForMaybe(MaybeData.success(userFullName))
      addCellForMaybe(MaybeData(activityLogEntry.getMessage), styles.wrapped)
      addCellForMaybe(MaybeData.nonEmptyString(itemText))
      addCellForMaybe(MaybeData.nonEmptyString(Type.valueOf(activityLogEntry.getTargetType).getName))
  }

  def getItemText(id: String): String = {
    val itemText = if(id.contains(Ids.TASK_PREFIX) && taskIdFrom(id).nonEmpty) {
      getTaskTitle(taskIdFrom(id))
    } else if(id.contains(Ids.PHASE_PREFIX) && phaseIdFrom(id).nonEmpty) {
      getPhaseTitle(phaseIdFrom(id))
    } else if(id.contains(Ids.RELEASE_PREFIX) && releaseIdFrom(id).nonEmpty) {
      getReleaseTitle(releaseIdFrom(id))
    } else {
      "Other"
    }
    itemText
  }

  def getPhaseTitle(id: String): String = {
    data.phases.get(id).getOrElse("")
  }

  def getTaskTitle(id: String): String = {
    data.tasks.get(id).getOrElse("")
  }

  def getTeamName(id: String): String = {
    data.teams.get(id).getOrElse("")
  }

  def getReleaseTitle(id: String): String = {
    data.releaseTitle.getOrElse("")
  }

}
