package com.xebialabs.xlrelease.risk.spring.config

import com.xebialabs.xlrelease.risk.spring.config.RestartableExecutorService.NOT_AVAILABLE_MSG
import com.xebialabs.xlrelease.service.{ExecutorServiceUtils, FeatureService}
import grizzled.slf4j.Logging

import java.util
import java.util.Collections
import java.util.concurrent._

class RestartableExecutorService(name: String, executorFactory: ExecutorServiceFactory) extends ExecutorService with FeatureService with Logging {
  @volatile
  private var enabled: Boolean = true
  private var _original: ExecutorService = executorFactory.create

  private def original(): Option[ExecutorService] = {
    if (enabled) {
      Some(_original)
    } else {
      None
    }
  }

  override def shutdown(): Unit = original.foreach(_.shutdown())

  override def shutdownNow(): util.List[Runnable] = original.map(_.shutdownNow()).getOrElse(Collections.emptyList())

  override def isShutdown: Boolean = original.map(_.isShutdown).getOrElse(true)

  override def isTerminated: Boolean = original.map(_.isTerminated).getOrElse(true)

  override def awaitTermination(timeout: Long, unit: TimeUnit): Boolean = original.map(_.awaitTermination(timeout, unit)).getOrElse(true)

  override def submit[T](task: Callable[T]): Future[T] = {
    original.map(_.submit(task)).getOrElse(throw new RejectedExecutionException(NOT_AVAILABLE_MSG))
  }

  override def submit[T](task: Runnable, result: T): Future[T] = {
    original.map(_.submit(task, result)).getOrElse(throw new RejectedExecutionException(NOT_AVAILABLE_MSG))
  }

  override def submit(task: Runnable): Future[_] = {
    original.map(_.submit(task)).getOrElse(throw new RejectedExecutionException(NOT_AVAILABLE_MSG))
  }

  override def invokeAll[T](tasks: util.Collection[_ <: Callable[T]]): util.List[Future[T]] = {
    original.map(_.invokeAll(tasks)).getOrElse(throw new RejectedExecutionException(NOT_AVAILABLE_MSG))
  }

  override def invokeAll[T](tasks: util.Collection[_ <: Callable[T]], timeout: Long, unit: TimeUnit): util.List[Future[T]] = {
    original.map(_.invokeAll(tasks, timeout, unit)).getOrElse(throw new RejectedExecutionException(NOT_AVAILABLE_MSG))
  }

  override def invokeAny[T](tasks: util.Collection[_ <: Callable[T]]): T = {
    original.map(_.invokeAny(tasks)).getOrElse(throw new RejectedExecutionException(NOT_AVAILABLE_MSG))
  }

  override def invokeAny[T](tasks: util.Collection[_ <: Callable[T]], timeout: Long, unit: TimeUnit): T = {
    original.map(_.invokeAny(tasks, timeout, unit)).getOrElse(throw new RejectedExecutionException(NOT_AVAILABLE_MSG))
  }

  override def execute(command: Runnable): Unit = {
    original.map(_.execute(command)).getOrElse(throw new RejectedExecutionException(NOT_AVAILABLE_MSG))
  }

  override def name(): String = name

  override def enable(): Unit = {
    logger.info(s"Enabling $name")
    synchronized {
      if (!isEnabled) {
        // create it again
        this._original = executorFactory.create
      }
      this.enabled = true
    }
  }

  override def disable(): Unit = {
    logger.info(s"Disabling $name")
    synchronized {
      original().foreach(ExecutorServiceUtils.shutdown(name, _))
      this.enabled = false
    }
  }

  override def isEnabled: Boolean = this.enabled
}

object RestartableExecutorService {
  val NOT_AVAILABLE_MSG = "Executor not available"
}


trait ExecutorServiceFactory {
  def create: ExecutorService
}