package com.xebialabs.xlrelease.risk.service

import com.xebialabs.xlrelease.domain.events._
import com.xebialabs.xlrelease.domain.{Release, Task}
import com.xebialabs.xlrelease.events.{EventBus, EventListener, Subscribe}
import com.xebialabs.xlrelease.risk.domain.progress.ReleaseProgress
import com.xebialabs.xlrelease.risk.repository.ProgressRepository
import grizzled.slf4j.Logging
import io.micrometer.core.annotation.Timed
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service

import scala.jdk.CollectionConverters._

@Service
@EventListener
class ProgressService @Autowired()(progressRepository: ProgressRepository, eventBus: EventBus) extends Logging {

  @Subscribe
  def releaseCreated(releaseCreatedEvent: ReleaseCreatedEvent): Unit = {
    // ReleaseCreatedEvent is not published through ActorSystem most of the time.
    // Since it is Subscribe, it will block the thread.
    updateReleaseProgress(releaseCreatedEvent.release)
  }

  @Subscribe
  def phaseClosedEvent(phaseClosedEvent: PhaseClosedEvent): Unit = {
    updateReleaseProgress(phaseClosedEvent.phase.getRelease)
  }

  @Subscribe
  def phaseDuplicated(phaseDuplicatedEvent: PhaseDuplicatedEvent): Unit = {
    updateReleaseProgress(phaseDuplicatedEvent.phaseDuplicate.getRelease)
  }

  @Subscribe
  def phaseDeleted(phaseDeletedEvent: PhaseDeletedEvent): Unit = {
    updateReleaseProgress(phaseDeletedEvent.phase.getRelease)
  }

  @Subscribe
  def taskSkipped(taskSkippedEvent: TaskSkippedEvent): Unit = {
    updateReleaseProgress(taskSkippedEvent.task.getRelease)
  }

  @Subscribe
  def taskCompleted(taskCompletedEvent: TaskCompletedEvent): Unit = {
    updateReleaseProgress(taskCompletedEvent.task.getRelease)
  }

  @Subscribe
  def taskCreated(taskCreatedEvent: TaskCreatedEvent): Unit = {
    updateReleaseProgress(taskCreatedEvent.task.getRelease)
  }

  @Subscribe
  def taskDuplicated(taskCopiedEvent: TaskCopiedEvent): Unit = {
    updateReleaseProgress(taskCopiedEvent.task.getRelease)
  }

  @Subscribe
  def taskCopiedFrom(taskCopiedFromEvent: TaskCopiedFromEvent): Unit = {
    updateReleaseProgress(taskCopiedFromEvent.task.getRelease)
  }

  @Subscribe
  def taskDeleted(taskDeletedEvent: TaskDeletedEvent): Unit = {
    updateReleaseProgress(taskDeletedEvent.task.getRelease)
  }

  @Subscribe
  def taskReopened(taskReopenedEvent: TaskReopenedEvent): Unit = {
    updateReleaseProgress(taskReopenedEvent.task.getRelease)
  }

  @Timed
  def updateReleaseProgress(release: Release): Unit = {
    if (!release.isTemplate) {
      val releaseProgressWithId = calculateReleaseProgress(release)
      storeProgress(releaseProgressWithId)
    }
  }

  private def storeProgress(releaseProgressWithReleaseId: ReleaseProgressWithReleaseId): ReleaseProgress = {
    val savedReleaseProgress = progressRepository.createOrUpdate(releaseProgressWithReleaseId.releaseProgress)
    eventBus.publish(ReleaseProgressCalculated(releaseProgressWithReleaseId.releaseId, savedReleaseProgress))
    savedReleaseProgress
  }

  private def calculateReleaseProgress(release: Release) = ReleaseProgressWithReleaseId(release.getId, calculateProgress(release))

  private def calculateProgress(release: Release): ReleaseProgress = {
    logger.debug(s"Calculating progress of release ${release.getId}")
    val allTasks = release.getAllTasks
    val totalTasks = allTasks.size
    val tasksCompleted = allTasks.asScala.count(isTaskCompleted)
    val releaseProgress = new ReleaseProgress
    releaseProgress.setId(release.getId + "/progress")
    releaseProgress.setTotalRemainingTasks(totalTasks - tasksCompleted)
    releaseProgress.setTotalTasks(totalTasks)
    val phasesProgress = release.getPhases.asScala.foldLeft(Map.empty[String, String]) {
      case (acc, phase) =>
        val totalPhaseTasks = phase.getAllTasks.size
        val tasksCompletedInPhase = phase.getAllTasks.asScala.count(isTaskCompleted)
        acc + (phase.getId -> s"$totalPhaseTasks;$tasksCompletedInPhase")
    }
    releaseProgress.setPhasesProgress(phasesProgress.asJava)
    releaseProgress
  }

  private def isTaskCompleted(task: Task) = {
    task.isDone || task.isDoneInAdvance
  }
}
