package com.xebialabs.xlrelease.risk.service

import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.events.{EventListener, SynchronizedSubscribe}
import com.xebialabs.xlrelease.repository.ConfigurationRepository
import com.xebialabs.xlrelease.risk.domain.RiskProfile
import com.xebialabs.xlrelease.risk.domain.events.RiskProfileUpdated
import grizzled.slf4j.Logging
import org.springframework.stereotype.Service

import java.util.concurrent.Executor


@Service
@EventListener
class RiskEventHandler(xlrConfig: XlrConfig,
                       configurationRepository: ConfigurationRepository,
                       riskService: RiskService,
                       riskCalculationBatchExecutor: Executor)
  extends Logging {

  @SynchronizedSubscribe
  def handleRiskProfileUpdate(event: RiskProfileUpdated): Unit = {
    if (xlrConfig.isRiskCalculationEnabled) {
      val riskProfile = event.riskProfile
      riskCalculationBatchExecutor.execute(() => {
        runCalculationsForRiskProfile(riskProfile)
      })
    }
  }

  private def runCalculationsForRiskProfile(riskProfile: RiskProfile): Unit = {
    logger.debug(s"Running risk calculations after Risk profile ${riskProfile.getId} update")
    configurationRepository.getReferenceReleases(riskProfile.getId).forEach(data => {
      riskService.calculateRiskAndUpdateRelease(data.id)
    })
    logger.debug(s"Finished running risk calculations after Risk profile ${riskProfile.getId} update")
  }

}
