package com.xebialabs.xlrelease.scm.connector;

import org.eclipse.jgit.api.Git;
import org.eclipse.jgit.api.LsRemoteCommand;
import org.eclipse.jgit.api.errors.GitAPIException;
import org.eclipse.jgit.lib.ObjectId;
import org.eclipse.jgit.lib.Ref;
import org.eclipse.jgit.transport.HttpTransport;

import java.util.Collection;

import static java.lang.String.format;
import static org.eclipse.jgit.lib.Constants.HEAD;
import static org.eclipse.jgit.lib.Constants.R_HEADS;

public class GitClient {

    static final ThreadLocal<GitConnectionSettings> currentGitConnectionSettings = new ThreadLocal<>();

    static {
        HttpTransport.setConnectionFactory(new XlrGitHttpConnectionFactory());
    }

    public GitClient(GitConnectionSettings gitConnectionSettings) {
        currentGitConnectionSettings.set(gitConnectionSettings);
    }

    public String getLatestRevision() throws GitAPIException {
        return getLatestRevisionObjectId().getName();
    }

    public ObjectId getLatestRevisionObjectId() throws GitAPIException {
        Collection<Ref> refs = fetchRemoteReferences();
        String branch = currentGitConnectionSettings.get().getBranch();
        return branch != null && !branch.isEmpty() ? findBranchSha1(refs, currentGitConnectionSettings.get().getBranch()) : findDefaultBranchSha1(refs);
    }

    public Collection<Ref> fetchRemoteReferences() throws GitAPIException {
        LsRemoteCommand lsRemoteCommand = Git.lsRemoteRepository().setRemote(currentGitConnectionSettings.get().getUrl()).setTags(false);
        return lsRemoteCommand.call();
    }

    private ObjectId findBranchSha1(Collection<Ref> refs, String branch) {
        return findSha1(refs, R_HEADS + branch);
    }

    private ObjectId findDefaultBranchSha1(Collection<Ref> refs) {
        return findSha1(refs, HEAD);
    }

    private ObjectId findSha1(Collection<Ref> refs, String input) {
        for (Ref ref : refs) {
            if (ref.getName().equals(input)) {
                return ref.getObjectId();
            }
        }
        throw new IllegalArgumentException(format("'%s' not found on %s", input, currentGitConnectionSettings.get().getUrl()));
    }

    public static void removeCurrentGitConnectionSettings() {
        currentGitConnectionSettings.remove();
    }

}
