package com.xebialabs.xlrelease.scm.connector;

import java.io.IOException;
import java.net.Proxy;
import java.net.URL;
import java.util.Collections;
import org.apache.commons.codec.binary.Base64;
import org.apache.http.HttpHost;
import org.apache.http.auth.AuthScope;
import org.apache.http.auth.Credentials;
import org.apache.http.auth.NTCredentials;
import org.apache.http.auth.UsernamePasswordCredentials;
import org.apache.http.client.CredentialsProvider;
import org.apache.http.client.HttpClient;
import org.apache.http.client.config.CookieSpecs;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.impl.client.BasicCredentialsProvider;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.message.BasicHeader;
import org.eclipse.jgit.transport.http.HttpConnection;
import org.eclipse.jgit.transport.http.HttpConnectionFactory;
import org.eclipse.jgit.transport.http.apache.HttpClientConnection;

import static java.lang.String.format;
import static java.nio.charset.StandardCharsets.UTF_8;
import static java.util.Collections.singletonList;
import static org.apache.http.HttpHeaders.AUTHORIZATION;
import static org.apache.http.client.config.AuthSchemes.NTLM;

public class XlrGitHttpConnectionFactory implements HttpConnectionFactory {

    XlrGitHttpConnectionFactory() {
    }

    @Override
    public HttpConnection create(final URL url) throws IOException {
        return new HttpClientConnection(url.toString(), null, createClient());
    }

    @Override
    public HttpConnection create(final URL url, final Proxy proxy) throws IOException {
        return new HttpClientConnection(url.toString(), null, createClient());
    }

    private HttpClient createClient() {
        final GitConnectionSettings gitConnectionSettings = GitClient.currentGitConnectionSettings.get();

        HttpClientBuilder clientBuilder = HttpClients.custom();
        RequestConfig.Builder configBuilder = RequestConfig.custom();
        configBuilder.setCookieSpec(CookieSpecs.STANDARD);
        CredentialsProvider credentialsProvider = new BasicCredentialsProvider();

        if (gitConnectionSettings.shouldUseAuthentication()) {
            switch (gitConnectionSettings.getAuthenticationMethod()) {
                case "Basic":
                    String basicAuthorization = Base64.encodeBase64String(format("%s:%s", gitConnectionSettings.getUsername(), gitConnectionSettings.getPassword()).getBytes(UTF_8));
                    clientBuilder.setDefaultHeaders(Collections.singletonList(new BasicHeader(AUTHORIZATION, format("Basic %s", basicAuthorization))));
                    break;
                case "Ntlm":
                    configBuilder.setTargetPreferredAuthSchemes(singletonList(NTLM));
                    credentialsProvider.setCredentials(AuthScope.ANY, new NTCredentials(gitConnectionSettings.getUsername(), gitConnectionSettings.getPassword(), null, gitConnectionSettings.getDomain()));
                    break;
                case "PAT":
                    String patAuthorization = Base64.encodeBase64String(format(":%s", gitConnectionSettings.getPassword()).getBytes(UTF_8));
                    clientBuilder.setDefaultHeaders(Collections.singletonList(new BasicHeader(AUTHORIZATION, format("Basic %s", patAuthorization))));
                    break;
            }
        }

        if (gitConnectionSettings.shouldUseProxy()) {
            clientBuilder.setProxy(new HttpHost(gitConnectionSettings.getProxyHost(), gitConnectionSettings.getProxyPort()));

            if (gitConnectionSettings.shouldUseProxyAuthentication()) {
                Credentials credentials;
                if (gitConnectionSettings.shouldUseNtlmProxyAuthentication()) {
                    credentials = new NTCredentials(gitConnectionSettings.getProxyUsername(), gitConnectionSettings.getProxyPassword(), null, gitConnectionSettings.getProxyDomain());
                } else {
                    credentials = new UsernamePasswordCredentials(gitConnectionSettings.getProxyUsername(), gitConnectionSettings.getProxyPassword());
                }
                AuthScope authScope = new AuthScope(gitConnectionSettings.getProxyHost(), gitConnectionSettings.getProxyPort());
                credentialsProvider.setCredentials(authScope, credentials);
            }
        }

        clientBuilder.setDefaultCredentialsProvider(credentialsProvider);
        clientBuilder.setDefaultRequestConfig(configBuilder.build());
        return clientBuilder.build();
    }
}
