package com.xebialabs.xlrelease.scm.connector

import com.xebialabs.xlrelease.domain.UserProfile
import com.xebialabs.xlrelease.domain.scm.connector.RestApiScmConnectorConfig
import com.xebialabs.xlrelease.domain.utils.ScmException
import com.xebialabs.xlrelease.scm.data.ValidatedCommitInfo

import scala.util.{Failure, Success, Try}

/**
 * Connects XLR templates to a SCM repository.
 */
trait ScmConnector extends AutoCloseable {
  // todo move this out of the trait
  // todo rename parameter
  def getIdentifier(uniqueString: String): String

  /**
   * Test whether this connector can establish a connection with the repository or not and cleans it up
   *
   * @return Success() if connection was successful, Failute(ex) if there was an error with establishing connection
   */
  def connectionStatus(): Try[Unit]

  /**
   * Check if the tag doesn't already exist in repository
   *
   * @param tag the tag to check
   * @return Success(true) if doesn't exist, Success(false) if tag exists, Failure if there was an error during check
   */
  protected def tagNotPresent(tag: String): Try[Boolean]

  def createNewVersion(blobs: ScmBlobs, commitInfo: ValidatedCommitInfo, user: UserProfile): Try[String] = {
    val result = connectionStatus() match {
      case Success(_) =>
        tagNotPresent(commitInfo.tag.refName).flatMap { notPresent =>
          if (notPresent) {
            commitAndTag(blobs, commitInfo, user)
          } else {
            Failure(ScmException(s"Tag '${commitInfo.tag.refName}' already exists"))
          }
        }
      case Failure(ex: ScmException) =>
        Failure(ScmException(ex.getMessage, ex, ex.statusCode))
      case Failure(ex) =>
        Failure(ScmException(ex.getMessage, ex))
    }
    result
  }

  /**
   * Create a commit and a tag, and push both to a remote repository
   *
   * @param blobs      The template DSL and attachments to commit
   * @param commitInfo Commit message and tag name
   * @param user       The user that created this commit
   * @return the commit SHA-1 if the commit was successful
   */
  protected def commitAndTag(blobs: ScmBlobs, commitInfo: ValidatedCommitInfo, user: UserProfile): Try[String]
}

abstract class RestApiScmConnector(config: RestApiScmConnectorConfig[_]) extends ScmConnector {
  override def close(): Unit = {
    // nothing to close
  }

  override def getIdentifier(uniqueString: String): String = {
    s"${config.restApiUrl}/${config.repository}/$uniqueString"
  }

  override def connectionStatus(): Try[Unit] = for {
    _ <- verifyRepositoryPath
    _ <- testRepository
  } yield ()

  protected def testRepository: Try[Unit]

  private def verifyRepositoryPath: Try[Unit] = {
    if (config.repository.split('/').length == 2) {
      Success(())
    } else {
      Failure(ScmException(s"Invalid repository format ${config.repository}"))
    }
  }
}
