package com.xebialabs.xlrelease.scm.data

import com.xebialabs.xlrelease.domain.utils.ScmException
import org.eclipse.jgit.lib.Repository
import org.springframework.util.StringUtils

import scala.util.{Failure, Success, Try}

sealed trait GitRef {
  def refType: String

  def refName: String

  override def toString: String = s"refs/$refType/$refName"
}

case class GitTag(refName: String) extends GitRef {
  override def refType: String = "tags"
}

sealed trait ValidatedCommitInfo {
  val tag: GitTag
  val message: String
}

object ValidatedCommitInfo {
  def create(version: String, message: String, tagPrefix: String): Try[ValidatedCommitInfo] = {
    create(version, message, GitTag(s"$tagPrefix/$version"))
  }

  def create(version: String, message: String, tagRef: GitTag): Try[ValidatedCommitInfo] = {
    if (!Repository.isValidRefName(tagRef.toString)) {
      Failure(ScmException(s"'${tagRef.refName}' is not a valid tag name"))
    } else {
      val validCommitMessage = if (StringUtils.hasText(message)) {
        message
      } else {
        version
      }

      Success(new ValidatedCommitInfo {
        override val tag: GitTag = tagRef
        override val message: String = validCommitMessage
      })
    }
  }

}
