import React from 'react';
import { ReactWrapper } from 'enzyme';
import { mountWithStoreAndTheme } from '../../../../../../../../../../../core/xlr-ui/tests/unit/testing-utils';
import { IMG_SRC } from '../../../constants';
import { folderExternalDeployments, FolderExternalDeploymentsState, initialState } from '../../../ducks/external-deployments.reducer';
import { BasicDiscovery } from './application-discovery-basic-component';
import { deployServerMock } from '../../../__mocks__/external-deployments.mocks';
import { DotAutoComplete } from '@digital-ai/dot-components';

const dispatch = jest.fn();
const { setLiveUpdate } = folderExternalDeployments.actions;
describe('Basic discovery component', () => {
    let wrapper: ReactWrapper;
    const onCancel = jest.fn();
    const onChange = jest.fn();
    const onSave = jest.fn();
    const defaultProps = {
        folder: { id: 'folderId', title: 'Folder-1' },
        isLoading: false,
        onCancel,
        onChange,
        onSave,
        servers: [deployServerMock],
        webhookSources: [],
    };
    const defaultState = {
        ...initialState,
    };

    const mountComponent = (state: FolderExternalDeploymentsState = defaultState) => {
        wrapper = mountWithStoreAndTheme(<BasicDiscovery {...defaultProps} />, dispatch, { folderExternalDeployments: state });
    };

    beforeEach(() => {
        mountComponent();
    });

    afterEach(() => {
        wrapper.unmount();
    });

    const searchDotCard = (selector: string) => wrapper.find(`.discovery ${selector}`);

    it('should contain title', () => {
        expect(searchDotCard(`h2`).text()).toStrictEqual('Discover applications');
    });

    it('should display argo server card', () => {
        expect(searchDotCard(`.server-cards .dot-card`).at(0)).toExist();
        expect(searchDotCard('.server-cards .dot-card .dot-card-header img').at(0).props().src).toStrictEqual(IMG_SRC.argoCd);
        expect(searchDotCard(`.server-cards .dot-card .dot-card-header span`).at(0).text()).toStrictEqual('ArgoCD');
        expect(searchDotCard(`.server-cards .dot-card .dot-card-header span`).at(1).text()).toStrictEqual('by Argo');
    });

    it('should show options for discovering argo servers', () => {
        expect(searchDotCard(`.server-cards .dot-card .dot-card-content button[data-testid="create-server-btn"]`)).toExist();
        expect(searchDotCard(`.server-cards .dot-card .dot-card-content button[data-testid="create-server-btn"]`).text()).toStrictEqual('Create server');
    });

    it('should display deploy server card', () => {
        expect(searchDotCard(`.server-cards .dot-card`).at(1)).toExist();
        expect(searchDotCard(`.server-cards .dot-card .dot-card-header img`).at(1).props().src).toStrictEqual(IMG_SRC.deploy);
        expect(searchDotCard(`.server-cards .dot-card .dot-card-header span`).at(2).text()).toStrictEqual('Deploy');
        expect(searchDotCard(`.server-cards .dot-card .dot-card-header span`).at(3).text()).toStrictEqual('by Digital.ai');
    });

    it('should show autocomplete for choosing deploy servers', () => {
        expect(wrapper.find(DotAutoComplete)).toExist();
        expect(wrapper.find(DotAutoComplete)).toHaveLength(1);
        expect(wrapper.find(DotAutoComplete).props().options).toStrictEqual([deployServerMock]);
    });

    it('should show button for creating deploy server', () => {
        expect(searchDotCard(`button[data-testid="discover-server-btn"]`)).toExist();
        expect(searchDotCard(`button[data-testid="discover-server-btn"]`).text()).toStrictEqual('Discover applications');
    });

    it('should disable discover applications button for deploy servers', () => {
        expect(searchDotCard(`button[data-testid="discover-server-btn"]`)).toBeDisabled();
    });

    it('should enable discover applications button for deploy servers', async () => {
        wrapper.find(DotAutoComplete).invoke('onChange')?.('' as never, deployServerMock, '');
        wrapper.update();
        expect(searchDotCard(`button[data-testid="discover-server-btn"]`)).not.toBeDisabled();
    });

    it('should enable discover applications button for deploy servers', async () => {
        wrapper.find(DotAutoComplete).invoke('onChange')?.('' as never, deployServerMock, '');
        wrapper.update();
        expect(searchDotCard(`.server-cards .dot-card .dot-card-content button[data-testid="discover-server-btn"]`)).not.toBeDisabled();
    });

    it('should handle discover applications button click', async () => {
        wrapper.find(DotAutoComplete).invoke('onChange')?.('' as never, deployServerMock, '');
        searchDotCard(`button[data-testid="discover-server-btn"]`).simulate('click');
        expect(dispatch).toBeCalledWith(setLiveUpdate({ folderId: 'folderId', serverId: deployServerMock.id! }));
    });

    it('should show cancel button', () => {
        expect(searchDotCard(`button[data-testid="discovery-cancel-btn"]`)).toExist();
        expect(searchDotCard(`button[data-testid="discovery-cancel-btn"]`).text()).toEqual('Cancel');
    });

    it('should handle cancel button click', () => {
        searchDotCard(`button[data-testid="discovery-cancel-btn"]`).simulate('click');
        expect(onCancel).toHaveBeenCalled();
    });
});
