import React, { useState, memo } from 'react';
import { useDispatch, useSelector } from 'react-redux';
import { ENDPOINT_TYPE } from '../../../constants';
import { DotAutoComplete, DotButton, DotCard, DotCardHeader, DotCardContent, DotThemeProvider } from '@digital-ai/dot-components';
import { getDiscoveryCardSubheader, getDiscoveryCardTitle, getServerIcon } from '../../../helper/utils';
import { CreateServerForm } from '../create-server/application-create-server-component';
import { DiscoverModal } from '../discover-modal/discover-modal-component';
import { Server, WebhookSource, WebhookSourceAutoConfigDetails } from '../../../external-deployment.types';
import { Folder } from '../../../../../../../../../../../core/xlr-ui/app/types';
import './application-discovery-basic-component.less';
import '../applications-discovery-component.less';
import { FilterApplications } from '../filter-applications/filter-applications-component';
import { folderExternalDeployments, FolderExternalDeploymentsState } from '../../../ducks/external-deployments.reducer';
import { getConfigDetailsSelector } from '../../../ducks/external-deployments.selectors';

export interface ApplicationDiscoveryBasicComponentProp {
    folder: Folder;
    isLoading: boolean;
    onCancel: () => void;
    onChange: () => void;
    onSave: () => void;
    servers: Array<Server>;
    webhookSources: Array<WebhookSource>;
}

const { deleteWebhookSource, setConfigDetails, setLiveUpdate } = folderExternalDeployments.actions;

const ApplicationDiscoveryBasicComponent = (props: ApplicationDiscoveryBasicComponentProp) => {
    const { isLoading, folder, onCancel, onSave, servers, webhookSources } = props;

    const webhookSourceAutoConfigDetails: WebhookSourceAutoConfigDetails | undefined = useSelector((state: FolderExternalDeploymentsState) =>
        getConfigDetailsSelector(state),
    );

    const [autoCreatedWebhook, setAutoCreatedWebhook] = useState<WebhookSource | undefined>(undefined);
    const [selectedDeployServer, setSelectedDeployServer] = useState<Server | undefined>(undefined);
    const [selectedArgoServer, setSelectedArgoServer] = useState<Server | undefined>(undefined);
    const [typeOfCreatedServer, setTypeOfCreatedServer] = useState<string | undefined>(undefined);
    const [discoverType, setDiscoverType] = useState<string | undefined>(undefined);
    const [selectedServer, setSelectedServer] = useState<Server | undefined>(undefined);

    const [isModalOpen, setIsModalOpen] = useState<boolean>(false);
    const [isFilterAppsOpen, setIsFilterAppsOpen] = useState<boolean>(false);

    const dispatch = useDispatch();

    const argoServers: Array<Server> = servers.filter((server: Server) => server.type === ENDPOINT_TYPE.argoCd);
    const deployServers: Array<Server> = servers.filter((server: Server) => server.type === ENDPOINT_TYPE.deploy);

    const updateSelectedServer = (server: Server, serverType: string) => {
        if (serverType === ENDPOINT_TYPE.deploy) setSelectedDeployServer(server);
        else if (serverType === ENDPOINT_TYPE.argoCd) setSelectedArgoServer(server);
        setDiscoverType(serverType);
    };

    const getSelectedServer = (serverType: string) => {
        if (serverType === ENDPOINT_TYPE.deploy) return selectedDeployServer;
        else if (serverType === ENDPOINT_TYPE.argoCd) return selectedArgoServer;
    };

    const checkIfSelected = (serverType: string) => {
        if (serverType === ENDPOINT_TYPE.deploy) return selectedDeployServer !== undefined;
        else if (serverType === ENDPOINT_TYPE.argoCd) return selectedArgoServer !== undefined;
    };

    const handleDiscoverApplication = (serverType: string) => {
        setSelectedServer(getSelectedServer(serverType));
        const serverId = getSelectedServer(serverType)?.id;

        if (serverId) {
            dispatch(setLiveUpdate({ folderId: folder.id, serverId }));
            setIsModalOpen(true);
        }
    };

    const onModalSubmit = () => {
        if (webhookSourceAutoConfigDetails?.webhookSourceId) {
            setAutoCreatedWebhook(webhookSources.find((webhookSource: WebhookSource) => webhookSource?.id === webhookSourceAutoConfigDetails?.webhookSourceId));
        }
        setIsModalOpen(false);
        setIsFilterAppsOpen(true);
        dispatch(setConfigDetails(undefined));
    };

    const closeModal = () => {
        cleanAutoCreatedWebhook();
        setIsModalOpen(false);
    };

    const filterBack = () => {
        cleanAutoCreatedWebhook();
        setIsFilterAppsOpen(false);
    };

    const filterCancel = () => {
        cleanAutoCreatedWebhook();
        onCancel();
    };

    const cleanAutoCreatedWebhook = () => {
        const webhookSourceId = autoCreatedWebhook ? autoCreatedWebhook.id : webhookSourceAutoConfigDetails?.webhookSourceId;
        if (webhookSourceId) {
            dispatch(deleteWebhookSource(webhookSourceId));
        }
        dispatch(setConfigDetails(undefined));
    };

    const generateServerCard = (serverType: string, serverOptions: Array<Server>) => {
        return (
            <DotCard>
                <DotCardHeader
                    avatar={<img alt={serverType} src={getServerIcon(serverType)} />}
                    className="card-header-inline"
                    subheader={getDiscoveryCardSubheader(serverType)}
                    subheaderSize="small"
                    title={getDiscoveryCardTitle(serverType)}
                    titleSize="small"
                />
                <DotCardContent>
                    {serverOptions.length === 0 && (
                        <DotButton data-testid="create-server-btn" className="discovery-button" onClick={() => setTypeOfCreatedServer(serverType)}>
                            Create server
                        </DotButton>
                    )}
                    {serverOptions.length > 0 && (
                        <DotAutoComplete
                            actionItem={{
                                iconId: 'add',
                                onClick: () => setTypeOfCreatedServer(serverType),
                                text: 'Create source',
                            }}
                            className="server-source"
                            freesolo={false}
                            inputId="server-source-input"
                            label="Choose server"
                            multiple={false}
                            onChange={(e, v) => updateSelectedServer(v as Server, serverType)}
                            options={serverOptions}
                        />
                    )}
                    {serverOptions.length > 0 && (
                        <DotButton
                            data-testid="discover-server-btn"
                            className="discovery-button"
                            disabled={!checkIfSelected(serverType)}
                            onClick={() => handleDiscoverApplication(serverType)}
                        >
                            Discover applications
                        </DotButton>
                    )}
                </DotCardContent>
                {webhookSourceAutoConfigDetails && (
                    <DiscoverModal
                        configureDetails={webhookSourceAutoConfigDetails}
                        isModalOpen={isModalOpen && !isLoading}
                        onModalClose={closeModal}
                        onModalSubmit={() => onModalSubmit()}
                        type={discoverType!}
                    />
                )}
            </DotCard>
        );
    };

    const generateServerCards = () => {
        return (
            <div className="discovery">
                <h2>Discover applications</h2>
                <div className="server-cards">
                    {generateServerCard(ENDPOINT_TYPE.argoCd, argoServers)}
                    {generateServerCard(ENDPOINT_TYPE.deploy, deployServers)}
                </div>
                <DotButton data-testid="discovery-cancel-btn" onClick={onCancel} type="text">
                    Cancel
                </DotButton>
            </div>
        );
    };

    const generateBasicView = () => {
        return (
            <>
                {!typeOfCreatedServer && generateServerCards()}
                {typeOfCreatedServer && (
                    <CreateServerForm closeForm={() => setTypeOfCreatedServer(undefined)} folder={folder} serverType={typeOfCreatedServer} />
                )}
            </>
        );
    };

    return (
        <DotThemeProvider>
            {!isFilterAppsOpen && generateBasicView()}
            {isFilterAppsOpen && (
                <FilterApplications
                    onCancel={filterCancel}
                    onFilterBack={filterBack}
                    onSave={onSave}
                    server={selectedServer}
                    webhookSource={autoCreatedWebhook as WebhookSource}
                />
            )}
        </DotThemeProvider>
    );
};

export const BasicDiscovery = memo(ApplicationDiscoveryBasicComponent);
