import React from 'react';
import { ReactWrapper } from 'enzyme';
import { mountWithStoreAndTheme } from '../../../../../../../../../../../core/xlr-ui/tests/unit/testing-utils';
import { ENDPOINT_TYPE, SERVER_AUTHENTICATION_OPTIONS } from '../../../constants';
import { DotAutoComplete, DotInputText, AutoCompleteValue } from '@digital-ai/dot-components';
import { folderExternalDeployments, FolderExternalDeploymentsState, initialState } from '../../../ducks/external-deployments.reducer';
import { CreateServerForm } from './application-create-server-component';
import { Server } from '../../../external-deployment.types';

const dispatch = jest.fn();
const { createServer } = folderExternalDeployments.actions;
describe('Create server component', () => {
    let wrapper: ReactWrapper;
    const closeForm = jest.fn();
    const defaultProps = {
        closeForm,
        folder: { id: 'folderId', title: 'Folder-1' },
        serverType: ENDPOINT_TYPE.deploy,
    };

    const defaultState = {
        ...initialState,
    };
    const mountComponent = (state: FolderExternalDeploymentsState = defaultState) => {
        wrapper = mountWithStoreAndTheme(<CreateServerForm {...defaultProps} />, dispatch, { folderExternalDeployments: state });
    };

    const server: Server = {
        id: null,
        authenticationMethod: SERVER_AUTHENTICATION_OPTIONS[0].title,
        folderId: 'folderId',
        title: 'test-title',
        type: ENDPOINT_TYPE.deploy,
        url: 'test-url',
    };

    const doChangeAutocomplete = (value: AutoCompleteValue) => {
        wrapper.find(DotAutoComplete).invoke('onChange')?.('' as never, value, '');
    };

    const doChangeInput = (wrapper: ReactWrapper, index: number, value: string) => {
        wrapper.find(DotInputText).at(index).invoke('onChange')?.({
            target: {
                value,
            },
        } as never);
    };

    beforeEach(() => {
        mountComponent();
    });

    afterEach(() => {
        wrapper.unmount();
    });

    it('should contain title', () => {
        expect(wrapper.find(`h1`)).toExist();
        expect(wrapper.find(`h1`).text()).toStrictEqual('New Deploy server');
    });

    it('should contain input for server title', () => {
        expect(wrapper.find(DotInputText).at(0).props().label).toStrictEqual('Title');
    });

    it('should contain input for server url', () => {
        expect(wrapper.find(DotInputText).at(1).props().label).toStrictEqual('URL');
    });

    it('should contain autocomplete for server authentication', () => {
        expect(wrapper.find(DotAutoComplete).exists).toBeTruthy();
        // @ts-ignore
        expect(wrapper.find(DotAutoComplete).props().options.length).toEqual(4);
    });

    it('should generate input field for basic auth', () => {
        doChangeAutocomplete(SERVER_AUTHENTICATION_OPTIONS[1]);
        expect(wrapper.find(DotInputText).at(2).props().label).toStrictEqual('Username');
        expect(wrapper.find(DotInputText).at(3).props().label).toStrictEqual('Password');
    });

    it('should generate input field for ntlm auth', () => {
        doChangeAutocomplete(SERVER_AUTHENTICATION_OPTIONS[2]);
        expect(wrapper.find(DotInputText).at(2).props().label).toStrictEqual('Username');
        expect(wrapper.find(DotInputText).at(3).props().label).toStrictEqual('Password');
        expect(wrapper.find(DotInputText).at(4).props().label).toStrictEqual('Domain');
    });

    it('should generate input field for oauth2 auth', () => {
        doChangeAutocomplete(SERVER_AUTHENTICATION_OPTIONS[3]);
        expect(wrapper.find(DotInputText).at(2).props().label).toStrictEqual('Username');
        expect(wrapper.find(DotInputText).at(3).props().label).toStrictEqual('Password');
        expect(wrapper.find(DotInputText).at(4).props().label).toStrictEqual('Access Token URL');
        expect(wrapper.find(DotInputText).at(5).props().label).toStrictEqual('Client ID');
        expect(wrapper.find(DotInputText).at(6).props().label).toStrictEqual('Client Secret');
        expect(wrapper.find(DotInputText).at(7).props().label).toStrictEqual('Scope');
    });

    it('should contain proxy fields', () => {
        doChangeAutocomplete(SERVER_AUTHENTICATION_OPTIONS[0]);
        expect(wrapper.find(DotInputText).at(2).props().label).toStrictEqual('Proxy Host');
        expect(wrapper.find(DotInputText).at(3).props().label).toStrictEqual('Proxy Port');
        expect(wrapper.find(DotInputText).at(4).props().label).toStrictEqual('Proxy Username');
        expect(wrapper.find(DotInputText).at(5).props().label).toStrictEqual('Proxy Password');
        expect(wrapper.find(DotInputText).at(6).props().label).toStrictEqual('Proxy Domain');
    });

    it("should display disabled 'Save' button", () => {
        expect(wrapper.find(`button[data-testid="save-server-btn"]`).exists).toBeTruthy();
        expect(wrapper.find(`button[data-testid="save-server-btn"]`).text()).toStrictEqual('Save');
        expect(wrapper.find(`button[data-testid="save-server-btn"]`).props().disabled).toStrictEqual(true);
    });

    it("should display enabled 'Save' button", () => {
        doChangeInput(wrapper, 0, 'test-title');
        doChangeInput(wrapper, 1, 'test-url');
        doChangeAutocomplete(SERVER_AUTHENTICATION_OPTIONS[0]);
        expect(wrapper.find(`button[data-testid="save-server-btn"]`).props().disabled).toStrictEqual(false);
    });

    it("should handle 'Save' button click", () => {
        doChangeInput(wrapper, 0, 'test-title');
        doChangeInput(wrapper, 1, 'test-url');
        doChangeAutocomplete(SERVER_AUTHENTICATION_OPTIONS[0]);
        wrapper.find(`button[data-testid="save-server-btn"]`).simulate('submit');
        expect(dispatch).toBeCalledWith(createServer(server));
    });
});
